/*
 * Decompiled with CFR 0.152.
 */
package com.urbanairship.messagecenter;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.VisibleForTesting;
import com.urbanairship.Logger;
import com.urbanairship.PreferenceDataStore;
import com.urbanairship.channel.AirshipChannel;
import com.urbanairship.config.AirshipRuntimeConfig;
import com.urbanairship.http.RequestException;
import com.urbanairship.http.Response;
import com.urbanairship.job.JobInfo;
import com.urbanairship.job.JobResult;
import com.urbanairship.json.JsonList;
import com.urbanairship.json.JsonValue;
import com.urbanairship.messagecenter.Inbox;
import com.urbanairship.messagecenter.InboxApiClient;
import com.urbanairship.messagecenter.MessageDao;
import com.urbanairship.messagecenter.MessageEntity;
import com.urbanairship.messagecenter.User;
import com.urbanairship.messagecenter.UserCredentials;
import com.urbanairship.util.UAStringUtil;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

class InboxJobHandler {
    static final String ACTION_RICH_PUSH_MESSAGES_UPDATE = "ACTION_RICH_PUSH_MESSAGES_UPDATE";
    static final String ACTION_SYNC_MESSAGE_STATE = "ACTION_SYNC_MESSAGE_STATE";
    static final String ACTION_RICH_PUSH_USER_UPDATE = "ACTION_RICH_PUSH_USER_UPDATE";
    static final String EXTRA_FORCEFULLY = "EXTRA_FORCEFULLY";
    static final String LAST_MESSAGE_REFRESH_TIME = "com.urbanairship.user.LAST_MESSAGE_REFRESH_TIME";
    private static final String LAST_UPDATE_TIME = "com.urbanairship.user.LAST_UPDATE_TIME";
    private static final long USER_UPDATE_INTERVAL_MS = 86400000L;
    private final MessageDao messageDao;
    private final User user;
    private final Inbox inbox;
    private final PreferenceDataStore dataStore;
    private final AirshipChannel channel;
    private final InboxApiClient inboxApiClient;

    InboxJobHandler(@NonNull Context context, @NonNull Inbox inbox, @NonNull User user, @NonNull AirshipChannel channel, @NonNull AirshipRuntimeConfig runtimeConfig, @NonNull PreferenceDataStore dataStore, @NonNull MessageDao messageDao) {
        this(inbox, user, channel, dataStore, messageDao, new InboxApiClient(runtimeConfig));
    }

    @VisibleForTesting
    InboxJobHandler(@NonNull Inbox inbox, @NonNull User user, @NonNull AirshipChannel channel, @NonNull PreferenceDataStore dataStore, @NonNull MessageDao messageDao, @NonNull InboxApiClient inboxApiClient) {
        this.inbox = inbox;
        this.user = user;
        this.channel = channel;
        this.dataStore = dataStore;
        this.messageDao = messageDao;
        this.inboxApiClient = inboxApiClient;
    }

    @RestrictTo(value={RestrictTo.Scope.LIBRARY_GROUP})
    void removeStoredData() {
        this.dataStore.remove(LAST_MESSAGE_REFRESH_TIME);
        this.dataStore.remove(LAST_UPDATE_TIME);
    }

    @NonNull
    JobResult performJob(@NonNull JobInfo jobInfo) {
        switch (jobInfo.getAction()) {
            case "ACTION_RICH_PUSH_USER_UPDATE": {
                this.onUpdateUser(jobInfo.getExtras().opt(EXTRA_FORCEFULLY).getBoolean(false));
                break;
            }
            case "ACTION_RICH_PUSH_MESSAGES_UPDATE": {
                this.onUpdateMessages();
                break;
            }
            case "ACTION_SYNC_MESSAGE_STATE": {
                this.onSyncMessages();
            }
        }
        return JobResult.SUCCESS;
    }

    private void onUpdateMessages() {
        if (!this.user.isUserCreated()) {
            Logger.debug((String)"User has not been created, canceling messages update", (Object[])new Object[0]);
            this.inbox.onUpdateMessagesFinished(false);
        } else {
            boolean success = this.updateMessages();
            this.inbox.refresh(true);
            this.inbox.onUpdateMessagesFinished(success);
            this.syncReadMessageState();
            this.syncDeletedMessageState();
        }
    }

    private void onSyncMessages() {
        this.syncReadMessageState();
        this.syncDeletedMessageState();
    }

    private void onUpdateUser(boolean forcefully) {
        long now;
        long lastUpdateTime;
        if (!forcefully && (lastUpdateTime = this.dataStore.getLong(LAST_UPDATE_TIME, 0L)) <= (now = System.currentTimeMillis()) && lastUpdateTime + 86400000L >= now) {
            return;
        }
        boolean success = !this.user.isUserCreated() ? this.createUser() : this.updateUser();
        this.user.onUserUpdated(success);
    }

    private boolean updateMessages() {
        Logger.info((String)"Refreshing inbox messages.", (Object[])new Object[0]);
        String channelId = this.channel.getId();
        if (UAStringUtil.isEmpty((String)channelId)) {
            Logger.verbose((String)"The channel ID does not exist.", (Object[])new Object[0]);
            return false;
        }
        Logger.verbose((String)"Fetching inbox messages.", (Object[])new Object[0]);
        try {
            Response<JsonList> response = this.inboxApiClient.fetchMessages(this.user, channelId, this.dataStore.getLong(LAST_MESSAGE_REFRESH_TIME, 0L));
            Logger.verbose((String)"Fetch inbox messages response: %s", (Object[])new Object[]{response});
            if (response.isSuccessful()) {
                JsonList result = (JsonList)response.getResult();
                Logger.info((String)"InboxJobHandler - Received %s inbox messages.", (Object[])new Object[]{((JsonList)response.getResult()).size()});
                this.updateInbox((JsonList)response.getResult());
                this.dataStore.put(LAST_MESSAGE_REFRESH_TIME, response.getLastModifiedTime());
                return true;
            }
            if (response.getStatus() == 304) {
                Logger.debug((String)"Inbox messages already up-to-date. ", (Object[])new Object[0]);
                return true;
            }
            Logger.debug((String)"Unable to update inbox messages %s.", (Object[])new Object[]{response});
            return false;
        }
        catch (RequestException e) {
            Logger.debug((Throwable)e, (String)"Update Messages failed.", (Object[])new Object[0]);
            return false;
        }
    }

    private void updateInbox(JsonList serverMessages) {
        ArrayList<JsonValue> messagesToInsert = new ArrayList<JsonValue>();
        HashSet<String> serverMessageIds = new HashSet<String>();
        for (JsonValue message : serverMessages) {
            if (!message.isJsonMap()) {
                Logger.error((String)"InboxJobHandler - Invalid message payload: %s", (Object[])new Object[]{message});
                continue;
            }
            String messageId = message.optMap().opt("message_id").getString();
            if (messageId == null) {
                Logger.error((String)"InboxJobHandler - Invalid message payload, missing message ID: %s", (Object[])new Object[]{message});
                continue;
            }
            serverMessageIds.add(messageId);
            MessageEntity messageEntity = MessageEntity.createMessageFromPayload(messageId, message);
            if (messageEntity == null) {
                Logger.error((String)"InboxJobHandler - Message Entity is null", (Object[])new Object[0]);
                continue;
            }
            if (this.messageDao.messageExists(messageEntity.messageId)) continue;
            messagesToInsert.add(message);
        }
        if (messagesToInsert.size() > 0) {
            this.messageDao.insertMessages(MessageEntity.createMessagesFromPayload(null, messagesToInsert));
        }
        List<String> deletedMessageIds = this.messageDao.getMessageIds();
        deletedMessageIds.removeAll(serverMessageIds);
        this.messageDao.deleteMessages(deletedMessageIds);
    }

    private void syncDeletedMessageState() {
        String channelId = this.channel.getId();
        if (UAStringUtil.isEmpty((String)channelId)) {
            return;
        }
        this.messageDao.deleteDuplicates();
        List<MessageEntity> messagesToUpdate = this.messageDao.getLocallyDeletedMessages();
        ArrayList<String> idsToDelete = new ArrayList<String>();
        ArrayList<JsonValue> reportings = new ArrayList<JsonValue>();
        for (MessageEntity message : messagesToUpdate) {
            if (message.getMessageReporting() == null) continue;
            reportings.add(message.getMessageReporting());
            idsToDelete.add(message.getMessageId());
        }
        if (idsToDelete.size() == 0) {
            return;
        }
        Logger.verbose((String)"Found %s messages to delete.", (Object[])new Object[]{idsToDelete.size()});
        try {
            Response<Void> response = this.inboxApiClient.syncDeletedMessageState(this.user, channelId, reportings);
            Logger.verbose((String)"Delete inbox messages response: %s", (Object[])new Object[]{response});
            if (response.getStatus() == 200) {
                this.messageDao.deleteMessages(idsToDelete);
            }
        }
        catch (RequestException e) {
            Logger.debug((Throwable)e, (String)"Deleted message state synchronize failed.", (Object[])new Object[0]);
        }
    }

    private void syncReadMessageState() {
        String channelId = this.channel.getId();
        if (UAStringUtil.isEmpty((String)channelId)) {
            return;
        }
        this.messageDao.deleteDuplicates();
        List<MessageEntity> messagesToUpdate = this.messageDao.getLocallyReadMessages();
        ArrayList<String> idsToUpdate = new ArrayList<String>();
        ArrayList<JsonValue> reportings = new ArrayList<JsonValue>();
        for (MessageEntity message : messagesToUpdate) {
            if (message.getMessageReporting() == null) continue;
            reportings.add(message.getMessageReporting());
            idsToUpdate.add(message.getMessageId());
        }
        if (idsToUpdate.isEmpty()) {
            return;
        }
        Logger.verbose((String)"Found %s messages to mark read.", (Object[])new Object[]{idsToUpdate.size()});
        try {
            Response<Void> response = this.inboxApiClient.syncReadMessageState(this.user, channelId, reportings);
            Logger.verbose((String)"Mark inbox messages read response: %s", (Object[])new Object[]{response});
            if (response.getStatus() == 200) {
                this.messageDao.markMessagesReadOrigin(idsToUpdate);
            }
        }
        catch (RequestException e) {
            Logger.debug((Throwable)e, (String)"Read message state synchronize failed.", (Object[])new Object[0]);
        }
    }

    private boolean createUser() {
        String channelId = this.channel.getId();
        if (UAStringUtil.isEmpty((String)channelId)) {
            Logger.debug((String)"No Channel. User will be created after channel registrations finishes.", (Object[])new Object[0]);
            return false;
        }
        try {
            Response<UserCredentials> response = this.inboxApiClient.createUser(channelId);
            if (response.isSuccessful()) {
                UserCredentials userCredentials = (UserCredentials)response.getResult();
                Logger.info((String)"InboxJobHandler - Created Rich Push user: %s", (Object[])new Object[]{userCredentials.getUsername()});
                this.dataStore.put(LAST_UPDATE_TIME, System.currentTimeMillis());
                this.dataStore.remove(LAST_MESSAGE_REFRESH_TIME);
                this.user.onCreated(userCredentials.getUsername(), userCredentials.getPassword(), channelId);
                return true;
            }
            Logger.debug((String)"Rich Push user creation failed: %s", (Object[])new Object[]{response});
            return false;
        }
        catch (RequestException e) {
            Logger.debug((Throwable)e, (String)"User creation failed.", (Object[])new Object[0]);
            return false;
        }
    }

    private boolean updateUser() {
        String channelId = this.channel.getId();
        if (UAStringUtil.isEmpty((String)channelId)) {
            Logger.debug((String)"No Channel. Skipping Rich Push user update.", (Object[])new Object[0]);
            return false;
        }
        try {
            Response<Void> response = this.inboxApiClient.updateUser(this.user, channelId);
            Logger.verbose((String)"Update Rich Push user response: %s", (Object[])new Object[]{response});
            int status = response.getStatus();
            if (status == 200) {
                Logger.info((String)"Rich Push user updated.", (Object[])new Object[0]);
                this.dataStore.put(LAST_UPDATE_TIME, System.currentTimeMillis());
                this.user.onUpdated(channelId);
                return true;
            }
            if (status == 401) {
                Logger.debug((String)"Re-creating Rich Push user.", (Object[])new Object[0]);
                this.dataStore.put(LAST_UPDATE_TIME, 0);
                return this.createUser();
            }
            this.dataStore.put(LAST_UPDATE_TIME, 0);
            return false;
        }
        catch (RequestException e) {
            Logger.debug((Throwable)e, (String)"User update failed.", (Object[])new Object[0]);
            return false;
        }
    }
}

