package com.usercentrics.sdk.core.time

import java.text.SimpleDateFormat
import java.util.*
import java.util.concurrent.TimeUnit

internal actual class DateTime {

    actual val year: Int by lazy { calendar.get(Calendar.YEAR) }
    actual val month: Int by lazy { calendar.get(Calendar.MONTH) + 1 }
    actual val day: Int by lazy { calendar.get(Calendar.DAY_OF_MONTH) }
    actual val hours: Int by lazy { calendar.get(Calendar.HOUR_OF_DAY) }
    actual val minutes: Int by lazy { calendar.get(Calendar.MINUTE) }
    actual val seconds: Int by lazy { calendar.get(Calendar.SECOND) }

    private val calendar: Calendar

    actual constructor() : this(now())
    actual constructor(timestamp: Long) : this(calendarFromTimestamp(timestamp))
    actual constructor(utcISOString: String) : this(calendarFromUtcISOString(utcISOString))

    constructor(calendar: Calendar) {
        this.calendar = calendar
    }

    actual fun timestamp(): Long {
        return calendar.time.time
    }

    actual fun addMonths(amount: Int): DateTime {
        return addField(Calendar.MONTH, amount)
    }

    actual fun addSeconds(amount: Int): DateTime {
        return addField(Calendar.SECOND, amount)
    }

    actual fun addDays(amount: Int): DateTime {
        return addField(Calendar.DAY_OF_MONTH, amount)
    }

    private fun addField(field: Int, amount: Int): DateTime {
        val newCalendar = Calendar.getInstance().apply {
            time = calendar.time
        }
        newCalendar.add(field, amount)
        return DateTime(newCalendar)
    }

    actual fun diffInDays(dateTime: DateTime): Int {
        val diff = this.timestamp() - dateTime.timestamp()
        return TimeUnit.DAYS.convert(diff, TimeUnit.MILLISECONDS).toInt()
    }

    actual fun formatLocalTimezone(): String {
        return localDateFormat.format(calendar.time)
    }

    actual fun atMidnight(): DateTime {
        val newCalendar = Calendar.getInstance(utcTimeZone).apply {
            time = calendar.time
        }

        newCalendar.set(Calendar.HOUR_OF_DAY, 0)
        newCalendar.set(Calendar.MINUTE, 0)
        newCalendar.set(Calendar.SECOND, 0)
        newCalendar.set(Calendar.MILLISECOND, 0)

        return DateTime(newCalendar)
    }

    actual operator fun compareTo(other: DateTime): Int {
        return this.timestamp().compareTo(other.timestamp())
    }

    actual override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as DateTime

        return timestamp() == other.timestamp()
    }

    actual override fun hashCode(): Int {
        return this.timestamp().hashCode()
    }

    actual companion object {

        actual var nowMocked: DateTime? = null
        private val utcTimeZone = TimeZone.getTimeZone("UTC")

        private val utcISODateFormat by lazy {
            SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", Locale.getDefault()).apply {
                timeZone = utcTimeZone
            }
        }

        private val localDateFormat by lazy {
            SimpleDateFormat("dd.MM.yyyy, HH:mm", Locale.getDefault())
        }

        private fun now(): Calendar {
            return nowMocked?.let { calendarFromTimestamp(it.timestamp()) } ?: Calendar.getInstance(utcTimeZone)
        }

        private fun calendarFromTimestamp(timestamp: Long): Calendar {
            return calendarFromDate(Date(timestamp))
        }

        private fun calendarFromDate(date: Date): Calendar {
            return Calendar.getInstance(utcTimeZone).apply {
                time = date
            }
        }

        private fun calendarFromUtcISOString(utcISOString: String): Calendar {
            try {
                val date = utcISODateFormat.parse(utcISOString)!!
                return calendarFromDate(date)
            } catch (ex: Exception) {
                throw DateParseException()
            }
        }
    }
}
