package com.usercentrics.sdk.services.api.http

import com.usercentrics.sdk.domain.api.http.HttpClient
import com.usercentrics.sdk.domain.api.http.HttpDisposable
import com.usercentrics.sdk.domain.api.http.HttpResponse
import com.usercentrics.sdk.models.api.HttpConstants
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import java.io.BufferedOutputStream
import java.io.FileNotFoundException
import java.io.InputStream
import java.io.OutputStream
import java.net.HttpURLConnection
import java.net.URL

internal class AndroidHttpClient(
    private val timeoutMillis: Long = 10_000L,
    private val dispatcher: Dispatcher
) : HttpClient {

    private val defaultCharset = Charsets.UTF_8
    private val jsonUtf8 = "application/json; charset=${defaultCharset.name()}"

    override fun get(url: String, headers: Map<String, String>): HttpResponse {
        return createConnection(url, headers).apply {
            requestMethod = "GET"
        }.use()
    }

    override fun get(url: String, headers: Map<String, String>, onSuccess: (HttpResponse) -> Unit, onError: (Throwable) -> Unit): HttpDisposable {
        val urlConnection = createConnection(url, headers).apply {
            requestMethod = "GET"
        }

        dispatcher.dispatch {
            onSuccess(urlConnection.use())
        }.onFailure {
            onError(it)
        }
        return HttpDisposable { urlConnection.disconnect() }
    }

    private fun mapResponseHeaders(headers: Map<String?, List<String?>>): Map<String, String> {
        return headers.map {
            // The framework is not serializing the headers from the response properly:
            // 1. Key/Value can be null
            // 2. Assume that the value that we want is the first one of the list of values
            (it.key?.lowercase() ?: "") to (it.value.firstOrNull() ?: "")
        }.toMap()
    }

    override fun post(url: String, headers: Map<String, String>, bodyData: String): String {
        return createConnection(url, headers).apply {
            doOutput = true
            requestMethod = "POST"
            setRequestProperty("Content-Type", jsonUtf8)
            writeStream(bodyData, outputStream)
        }.use().body
    }

    private fun createConnection(rawUrl: String, headers: Map<String, String>): HttpURLConnection {
        val url = URL(rawUrl)
        return (url.openConnection() as HttpURLConnection).apply {
            connectTimeout = timeoutMillis.toInt()
            readTimeout = timeoutMillis.toInt()
            setHeaders(headers, this)
            setRequestProperty("Accept", jsonUtf8)
        }
    }

    private fun setHeaders(headers: Map<String, String>, connection: HttpURLConnection) {
        for (entry in headers.entries) {
            connection.setRequestProperty(entry.key, entry.value)
        }
    }

    private fun HttpURLConnection.use(): HttpResponse {
        return try {
            connect()
            HttpResponse(
                headers = mapResponseHeaders(this.headerFields),
                statusCode = this.responseCode,
                body = parseResponseBody(this.inputStreamOrErrorStream().readBytes())
            )
        } catch (notFoundEx: FileNotFoundException) {
            HttpResponse(statusCode = HttpConstants.FORBIDDEN_STATUS_CODE)
        } finally {
            runCatching { this.inputStreamOrErrorStream().close() }
            runCatching { disconnect() }
        }
    }

    private fun HttpURLConnection.inputStreamOrErrorStream(): InputStream {
        return if (responseCode < 400) this.inputStream else this.errorStream
    }

    private fun parseResponseBody(responseBody: ByteArray): String {
        return String(responseBody)
    }

    private fun writeStream(bodyData: String, outputStream: OutputStream?) {
        val stream = BufferedOutputStream(outputStream)
        stream.write(bodyData.toByteArray(defaultCharset))
        stream.flush()
        stream.close()
        outputStream?.close()
    }
}
