package com.usercentrics.sdk.services.deviceStorage

import android.content.SharedPreferences

internal class SharedPreferencesKeyValueStorage(
    private val sharedPreferences: SharedPreferences
) : KeyValueStorage {

    override fun deleteKey(key: String) {
        sharedPreferences.edit().remove(key).apply()
    }

    override fun getString(key: String, defaultValue: String?): String? {
        return sharedPreferences.getString(key, defaultValue)
    }

    override fun getNumber(key: String, defaultValue: Int): Int {
        return sharedPreferences.getInt(key, defaultValue)
    }

    override fun getLong(key: String, defaultValue: Long): Long {
        return sharedPreferences.getLong(key, defaultValue)
    }

    override fun put(key: String, value: String) {
        sharedPreferences.edit().putString(key, value).apply()
    }

    override fun put(key: String, value: Int) {
        sharedPreferences.edit().putInt(key, value).apply()
    }

    override fun put(key: String, value: Long) {
        sharedPreferences.edit().putLong(key, value).apply()
    }

    override fun hasKey(key: String): Boolean {
        return sharedPreferences.contains(key)
    }

    override fun purgeStorage() {}

    @Suppress("INAPPLICABLE_JVM_NAME")
    @JvmName("putStringMap")
    override fun put(values: Map<String, String>) {
        putValuesMap(values)
    }

    @Suppress("INAPPLICABLE_JVM_NAME")
    @JvmName("putIntegerMap")
    override fun put(values: Map<String, Int>) {
        putValuesMap(values)
    }

    override fun putValuesMap(values: Map<String, Any>) {
        val editor = sharedPreferences.edit()
        for (entry in values) {
            when (val value = entry.value) {
                is String -> editor.putString(entry.key, value)
                is Int -> editor.putInt(entry.key, value)
            }
        }
        editor.apply()
    }

    override fun getStringWithKeyStartingWith(pattern: String): Map<String, String> {
        val result = mutableMapOf<String, String>()
        for ((key, value) in sharedPreferences.all) {
            if (key.startsWith(pattern) && value is String) {
                result[key] = value
            }
        }
        return result
    }

    override fun deleteKeysThatDoNotMatch(pattern: String, values: Set<String>) {
        val editor = sharedPreferences.edit()

        for (key in sharedPreferences.all.keys) {
            if (key.startsWith(pattern) && !values.contains(key.removePrefix(pattern))) {
                editor.remove(key)
            }
        }
        editor.apply()
    }

    override fun deleteAll(exceptions: List<String>) {
        val editor = sharedPreferences.edit()
        for (key in sharedPreferences.all.keys) {
            if (key in exceptions) {
                continue
            }
            editor.remove(key)
        }
        editor.apply()
    }
}
