package com.usercentrics.ccpa

import kotlinx.serialization.Serializable

@Serializable
data class CCPAData(
    /** CCPA version. */
    val version: Int,
    /** True if the notice was given. False if not. */
    val noticeGiven: Boolean?,
    /** True if the user opted out the consents, so the user denies the services. False if not, so the user accepts the services. */
    var optedOut: Boolean?,
    /** Limited Service Provider Agreement Covered Transaction. */
    val lspact: Boolean?,
) {

    /** Encoded USP String. */
    @Suppress("DEPRECATION")
    val uspString: String
        get() = toUSPString()

    @Deprecated("Please, use the field uspString directly")
    fun toUSPString() =
        StringBuilder()
            .append(version)
            .append(noticeGiven?.toYesOrNo() ?: '-')
            .append(optedOut?.toYesOrNo() ?: '-')
            .append(lspact?.toYesOrNo() ?: '-')
            .toString()

    companion object {
        private const val expectedLength = 4
        private const val versionPosition = 0
        private const val noticeGivenPosition = 1
        private const val optedOutPosition = 2
        private const val lspactPosition = 3

        fun fromCCPAString(ccpaString: String): CCPAData {
            if (ccpaString.length != expectedLength) {
                throw CCPAException.parseString(ccpaString)
            }
            try {
                return CCPAData(
                    version = ccpaString[versionPosition].toString().toInt(),
                    noticeGiven = ccpaString[noticeGivenPosition].yesOrNoToBoolean(),
                    optedOut = ccpaString[optedOutPosition].yesOrNoToBoolean(),
                    lspact = ccpaString[lspactPosition].yesOrNoToBoolean(),
                )
            } catch (e: IllegalArgumentException) {
                throw CCPAException.parseString(ccpaString, e)
            }
        }
    }
}

private fun Boolean.toYesOrNo() = if (this) 'Y' else 'N'
private fun Char.yesOrNoToBoolean(): Boolean? {
    return when (this) {
        'N' -> false
        'n' -> false
        'Y' -> true
        'y' -> true
        else -> null
    }
}
