package com.usercentrics.sdk

import com.usercentrics.sdk.core.application.Application
import com.usercentrics.sdk.core.application.UsercentricsApplication
import com.usercentrics.sdk.errors.*
import com.usercentrics.sdk.models.api.ApiConstants
import kotlinx.coroutines.job
import kotlin.concurrent.Volatile
import kotlin.coroutines.coroutineContext

@Suppress("RedundantSetter", "ObjectPropertyName")
internal object UsercentricsInternal {

    private val isReadyObservable: Observable<Result<UsercentricsSDK>> = Observable()

    private var isInitializing: Boolean = false
        private set
    private var onOngoingInitializationFinish: (() -> Unit)? = null
        private set

    private val application: Application?
        get() = UsercentricsApplication.instance?.application

    @Volatile
    internal var _instance: UsercentricsSDK? = null
        private set

    val instance: UsercentricsSDK
        get() {
            return when (val state = UsercentricsInstanceState.from(_instance, isReadyObservable.value)) {
                is UsercentricsInstanceState.Invalid -> throw state.cause
                is UsercentricsInstanceState.Valid -> state.value
            }
        }

    fun isReady(onSuccess: (UsercentricsReadyStatus) -> Unit, onFailure: (UsercentricsError) -> Unit) {
        isReadyObservable.subscribe { result ->
            result.onSuccess {
                // FIXME: decide a good approach to do this

                val readyStatusResult = kotlin.runCatching { it.readyStatus() }
                readyStatusResult.onSuccess {
                    onSuccess(it)
                }.onFailure {
                    onFailure(UsercentricsError(UsercentricsException(message = it.message ?: "", cause = it)))
                }

            }.onFailure {
                onFailure((it as UsercentricsException).asError())
            }
        }
    }

    fun initialize(options: UsercentricsOptions, context: UsercentricsContext?) {
        if (!isInitializing) {
            doInitialize(options, context)
            return
        }

        onOngoingInitializationFinish = {
            application?.logger?.warning("Initialize is being invoked more than once, make sure this is the intended behaviour.")
            doInitialize(options, context)
        }
    }

    private fun doInitialize(options: UsercentricsOptions, context: UsercentricsContext?) {
        if (_instance != null) {
            resetToInitializeAgain()
        }
        isInitializing = true

        var optionsCopy: UsercentricsOptions = options.copy()
        try {
            optionsCopy = validateOptions(optionsCopy)
        } catch (ex: Exception) {
            finishInitialization(Result.failure(ex))
            return
        }

        val application = bootApplication(optionsCopy, context)

        val usercentrics = usercentricsProvider.provide(application, optionsCopy, context)
        _instance = usercentrics

        initializeSDKOnline(usercentrics, application, options.initTimeoutMillis)
    }

    private fun validateOptions(options: UsercentricsOptions): UsercentricsOptions {
        val isInvalidSelfHosted = options.isSelfHostedConfigurationInvalid()

        val hasSettingsId = options.settingsId.isNotBlank()
        val hasRuleSetId = options.ruleSetId.isNotBlank()
        val hasExactlyOne = hasSettingsId xor hasRuleSetId

        when {
            !hasExactlyOne -> throw InvalidIdException()
            isInvalidSelfHosted -> throw UsercentricsException("Defined self hosting domains are not valid. Please validate them!")
        }
        return fixTimeoutValuesIfNeeded(options)
    }

    private fun fixTimeoutValuesIfNeeded(options: UsercentricsOptions): UsercentricsOptions {
        // FIXME: think about replacing this to the minimum timeout
        if (options.timeoutMillis <= 0) {
            options.timeoutMillis = ApiConstants.DEFAULT_TIMEOUT_MILLIS
        }

        if (options.initTimeoutMillis < ApiConstants.MINIMUM_TIMEOUT_MILLIS) {
            options.initTimeoutMillis = ApiConstants.MINIMUM_TIMEOUT_MILLIS
        }
        return options
    }

    private fun bootApplication(options: UsercentricsOptions, context: UsercentricsContext?): Application {
        UsercentricsApplication.setInitialValues(options, context)
        return UsercentricsApplication.provide().also {
            it.boot()
        }
    }

    private fun initializeSDKOnline(usercentrics: UsercentricsSDK, application: Application, timeout: Long) {
        val cacheStorage = application.etagCacheStorage.value
        val dispatcher = application.dispatcher

        dispatcher.dispatchWithTimeout(timeout) {
            cacheStorage.saveOfflineStaging()

            val initializeResult = usercentrics.initialize(offlineMode = false)
            val cancelled = coroutineContext.job.isCancelled
            if (initializeResult.isSuccess && !cancelled) {
                finishInitialization(Result.success(usercentrics))
                cacheStorage.removeOfflineStaging()
                return@dispatchWithTimeout
            }
            onFailureInitializingSDKOnline(wrapAsUsercentricsException(initializeResult.exceptionOrNull()))
        }.onFailure {
            onFailureInitializingSDKOnline(wrapAsUsercentricsException(it))
        }
    }

    private fun wrapAsUsercentricsException(throwable: Throwable?): UsercentricsException =
        when (throwable) {
            is UsercentricsException -> throwable
            else -> UsercentricsException(
                message = throwable?.message?.takeIf { it.isNotBlank() } ?: throwable?.toString() ?: "Unknown error",
                cause = throwable
            )
        }

    private fun onFailureInitializingSDKOnline(exception: UsercentricsException) {
        val application = application ?: return

        application.logger.warning("Usercentrics SDK was not able to initialize online, let's try to initialize offline", exception)

        application.dispatcher.dispatch {
            application.etagCacheStorage.value.restoreOfflineStaging()
            initializeSDKOffline(exception)
        }
    }

    private suspend fun initializeSDKOffline(initializeOnlineError: UsercentricsException) {
        val usercentrics = _instance ?: return

        val initializeResult = usercentrics.initialize(offlineMode = true)
        if (initializeResult.isSuccess) {
            finishInitialization(Result.success(usercentrics))
            return
        }

        onFailureInitializingSDKOffline(
            initializeOnlineError = initializeOnlineError,
            offlineException = UsercentricsException(message = "", cause = initializeResult.exceptionOrNull())
        )
    }

    private fun onFailureInitializingSDKOffline(initializeOnlineError: UsercentricsException, offlineException: UsercentricsException) {
        application?.logger?.warning(
            "Usercentrics SDK was not able to initialize offline, cannot initialize, please make sure the internet connection is fine and retry",
            offlineException
        )

        finishInitialization(Result.failure(InitializationFailedException(initializeOnlineError)))
    }

    private fun finishInitialization(result: Result<UsercentricsSDK>) {
        if (result.isSuccess) {
            application?.logger?.debug("Usercentrics SDK is fully initialized")
        }

        val onFinishCallback = onOngoingInitializationFinish
        onOngoingInitializationFinish = null
        isInitializing = false

        if (onFinishCallback != null) {
            onFinishCallback.invoke()
            return
        }
        isReadyObservable.set(result)

        application?.dispatcher?.dispatchMain {
            isReadyObservable.emit(result)
        }
    }

    fun reset() {
        UsercentricsApplication.tearDown(clearStorage = true)
        UsercentricsEvent.tearDown()
        isReadyObservable.disposeAll()

        _instance = null
        isInitializing = false
        onOngoingInitializationFinish = null
    }

    private fun resetToInitializeAgain() {
        UsercentricsApplication.tearDown(clearStorage = false)
        UsercentricsEvent.tearDown()
        isReadyObservable.disposeValue()

        _instance = null
    }
}

private sealed class UsercentricsInstanceState {

    companion object {

        fun from(instanceState: UsercentricsSDK?, isReadyState: Result<*>?): UsercentricsInstanceState {
            val initializationFailure = isReadyState?.exceptionOrNull()

            return when {
                initializationFailure != null -> {
                    Invalid(initializationFailure)
                }

                instanceState == null -> {
                    Invalid(NotInitializedException())
                }

                isReadyState?.isSuccess != true -> {
                    Invalid(NotReadyException())
                }

                else -> {
                    Valid(instanceState)
                }
            }
        }
    }

    class Valid(val value: UsercentricsSDK) : UsercentricsInstanceState()

    class Invalid(val cause: Throwable) : UsercentricsInstanceState()
}
