package com.usercentrics.sdk

import com.usercentrics.sdk.extensions.sortedAlphaBy
import com.usercentrics.sdk.models.settings.LegacyService
import com.usercentrics.sdk.models.settings.PredefinedUIDecision
import com.usercentrics.sdk.services.tcf.interfaces.*
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory

/**
 * @suppress
 * */
internal data class PurposeProps(
    val checked: Boolean,
    val legitimateInterestChecked: Boolean,
    val purpose: TCFPurpose
)

/**
 * @suppress
 * */

internal data class SpecialPurposeProps(
    val checked: Boolean,
    val legitimateInterestChecked: Boolean,
    val purpose: TCFSpecialPurpose
)

/**
 * @suppress
 * */
internal data class StackProps(
    val checked: Boolean,
    val stack: TCFStack
)

/**
 * @suppress
 * */
internal data class SpecialFeatureProps(
    val checked: Boolean,
    val specialFeature: TCFSpecialFeature
)

/**
 * @suppress
 * */
internal data class CategoryProps(
    val category: UsercentricsCategory,
    val checked: Boolean,
    val services: List<LegacyService>,
)

/**
 * @suppress
 * */
internal data class VendorProps(
    val checked: Boolean,
    val legitimateInterestChecked: Boolean,
    val vendor: TCFVendor
)

/**
 * @suppress
 * */
internal class UsercentricsMaps {
    companion object {
        fun mapPurposes(tcfData: TCFData): List<PurposeProps> {
            val sortedPurposes: List<TCFPurpose> = tcfData.purposes.sortedBy { purpose ->
                purpose.id
            }
            val mappedPurposes: MutableList<PurposeProps> = mutableListOf()
            sortedPurposes.forEach { purpose ->
                mappedPurposes.add(
                    PurposeProps(
                        checked = purpose.consent ?: PredefinedUIDecision.DEFAULT_CONSENT_VALUE,
                        legitimateInterestChecked = purpose.legitimateInterestConsent ?: PredefinedUIDecision.DEFAULT_LEGITIMATE_INTEREST_VALUE,
                        purpose = purpose
                    )
                )
            }
            return mappedPurposes.toList()
        }

        fun mapSpecialPurposes(tcfData: TCFData): List<SpecialPurposeProps> {
            val sortedPurposes: List<TCFSpecialPurpose> = tcfData.specialPurposes.sortedBy { purpose ->
                purpose.id
            }
            val mappedPurposes: MutableList<SpecialPurposeProps> = mutableListOf()
            sortedPurposes.forEach { purpose ->
                mappedPurposes.add(
                    SpecialPurposeProps(
                        checked = PredefinedUIDecision.DEFAULT_CONSENT_VALUE,
                        legitimateInterestChecked = PredefinedUIDecision.DEFAULT_LEGITIMATE_INTEREST_VALUE,
                        purpose = purpose
                    )
                )
            }
            return mappedPurposes.toList()
        }

        fun mapStacks(tcfData: TCFData): List<StackProps> {
            val sortedStacks: List<TCFStack> = tcfData.stacks.sortedBy { stack ->
                stack.id
            }
            val mappedStacks: MutableList<StackProps> = mutableListOf()
            sortedStacks.forEach { stack ->
                val purposes = tcfData.purposes.filter { purpose ->
                    stack.purposeIds.contains(purpose.id)
                }
                val specialFeatures = tcfData.specialFeatures.filter { specialFeature ->
                    stack.specialFeatureIds.contains(specialFeature.id)
                }
                // A stack is checked if SOME containing purposes or specialFeatures are checked
                val checked = purposes.any { purpose ->
                    purpose.consent == true
                } || specialFeatures.any { specialFeature ->
                    specialFeature.consent == true
                }
                mappedStacks.add(
                    StackProps(
                        checked = checked,
                        stack = stack
                    )
                )
            }
            return mappedStacks.toList()
        }

        fun mapSpecialFeatures(tcfData: TCFData): List<SpecialFeatureProps> {
            val sortedSpecialFeatures: List<TCFSpecialFeature> =
                tcfData.specialFeatures.sortedBy { specialFeature ->
                    specialFeature.id
                }
            val mappedSpecialFeatures: MutableList<SpecialFeatureProps> = mutableListOf()
            sortedSpecialFeatures.forEach { specialFeature ->
                mappedSpecialFeatures.add(
                    SpecialFeatureProps(
                        checked = specialFeature.consent ?: PredefinedUIDecision.DEFAULT_CONSENT_VALUE,
                        specialFeature = specialFeature
                    )
                )
            }
            return mappedSpecialFeatures.toList()
        }

        private fun getCategoryConsent(category: UsercentricsCategory, services: List<LegacyService>): Boolean {
            if (category.isEssential) {
                return true
            }
            return services.any { service ->
                service.consent.status
            }
        }

        fun mapCategories(categories: List<UsercentricsCategory>, services: List<LegacyService>): List<CategoryProps> {
            return categories
                .filter { !it.isHidden }
                .map { category ->
                    val categoryServices = services.filter { it.categorySlug == category.categorySlug }
                    val checked = getCategoryConsent(category, categoryServices)
                    CategoryProps(
                        checked = checked,
                        category = category,
                        services = categoryServices,
                    )
                }
                .filter { it.services.isNotEmpty() }
        }

        fun mapVendors(tcfData: TCFData): List<VendorProps> {
            val sortedVendors: List<TCFVendor> = tcfData.vendors.sortedAlphaBy { it.name }
            val mappedVendors: MutableList<VendorProps> = mutableListOf()
            sortedVendors.forEach { vendor ->
                mappedVendors.add(
                    VendorProps(
                        checked = vendor.consent ?: PredefinedUIDecision.DEFAULT_CONSENT_VALUE,
                        legitimateInterestChecked = vendor.legitimateInterestConsent ?: PredefinedUIDecision.DEFAULT_LEGITIMATE_INTEREST_VALUE,
                        vendor = vendor
                    )
                )
            }
            return mappedVendors
        }
    }
}
