package com.usercentrics.sdk

import com.usercentrics.ccpa.CCPAData
import com.usercentrics.sdk.core.application.Application
import com.usercentrics.sdk.errors.*
import com.usercentrics.sdk.mediation.data.ConsentMediationPayload
import com.usercentrics.sdk.mediation.data.TCFConsentPayload
import com.usercentrics.sdk.models.ccpa.CCPAErrors
import com.usercentrics.sdk.models.common.*
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.predefinedUI.PredefinedUIConsentManagerImpl
import com.usercentrics.sdk.services.tcf.TCFDecisionUILayer
import com.usercentrics.sdk.services.tcf.interfaces.AdTechProviderDecision
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.services.tcf.interfaces.TCFUserDecisions
import com.usercentrics.sdk.ui.PredefinedUIApplicationManager
import com.usercentrics.sdk.ui.PredefinedUIFactoryHolder
import com.usercentrics.sdk.v2.banner.service.BannerViewDataServiceImpl
import com.usercentrics.sdk.v2.settings.data.UsercentricsService
import com.usercentrics.sdk.v2.settings.data.VariantsSettings

internal class UsercentricsSDKImpl(
    private val application: Application,
    internal val options: UsercentricsOptions,
) : UsercentricsSDK() {

    companion object {
        const val setCmpIdError = "To set the CMP ID you *must* have the TCF settings enabled"
        fun customUITCFError(operation: String) = "You *must* have the TCF settings enabled to do this operation: $operation"
    }

    private var activeControllerId = ""

    private val isTCFEnabled: Boolean
        get() = application.settingsInstance.value.isTCFEnabled()

    private val tcfInstance
        get() = application.tcfInstance.value

    override suspend fun initialize(offlineMode: Boolean): Result<Unit> {
        application.networkStrategy.value.set(offlineMode)

        val settingsOrchestrator = application.settingsOrchestrator.value
        try {
            settingsOrchestrator.boot(options)
        } catch (ex: UsercentricsException) {
            return Result.failure(ex)
        }

        initializeControllerId()

        var initializeResult: Result<Unit>? = null
        try {
            initializeResult = settingsOrchestrator.coldInitialize(controllerId = activeControllerId)
        } finally {
            return finalizeInitializationRegardlessOfCancellation(initializeResult)
        }
    }

    private fun finalizeInitializationRegardlessOfCancellation(coldInitializeResult: Result<Unit>?): Result<Unit> {
        if (coldInitializeResult?.exceptionOrNull() != null) {
            return coldInitializeResult
        }

        if (isTCFEnabled) {
            // FIXES concurrent modification exception
            tcfInstance.getTCFData()
        }

        logConsentMediationInitialState()
        runMediationAfterInitialize()

        setupABTestingIfNeeded()

        return Result.success(Unit)
    }

    // TODO: refactor to have one controllerId key
    private fun initializeControllerId() {
        val storedControllerId = application.storageInstance.value.getControllerId()
        if (storedControllerId.isNotBlank()) {
            activeControllerId = storedControllerId
        }
    }

    override fun shouldCollectConsent(): Boolean {
        // FIXME: decide a good approach to do this

        val result = kotlin.runCatching {
            val initialView = application.initialValuesStrategy.value.resolveInitialView()
            initialView != InitialView.NONE
        }
        return result.getOrNull() ?: false
    }

    override fun getConsents(): List<UsercentricsServiceConsent> {
        return application.settingsInstance.value.getSettings().services.map { it.mapConsent() }
    }

    override fun getCMPData(): UsercentricsCMPData {
        return UsercentricsCMPData(
            settings = application.settingsService.settings!!.data,
            services = getServices(),
            legalBasis = application.translationService.translations!!,
            activeVariant = application.initialValuesStrategy.value.variant!!,
            userLocation = application.locationService.value.location
        )
    }

    private fun getServices(): List<UsercentricsService> {
        return application.settingsService.settings!!.services
    }

    override fun getControllerId(): String {
        var result = activeControllerId
        if (result.isBlank()) {
            result = application.settingsInstance.value.getSettings().controllerId
        }
        return result
    }

    override fun restoreUserSession(controllerId: String, onSuccess: (UsercentricsReadyStatus) -> Unit, onFailure: (UsercentricsError) -> Unit) {
        val onSuccessCallback = {
            application.dispatcher.dispatchMain {
                application.logger.debug("Restore User Session finished with success")

                onSuccess(readyStatus())
            }
        }
        val onError = { exception: UsercentricsException ->
            application.dispatcher.dispatchMain {
                val error = exception.asError()
                application.logger.error(error)

                onFailure(error)
            }
        }

        application.dispatcher.dispatch {
            assertNotUIThread()

            val settings = application.settingsService.settings?.data
            val xDeviceFlag = settings?.consentXDevice

            val variant = application.initialValuesStrategy.value.variant

            when {
                xDeviceFlag == null || variant == null -> onError(NotReadyException())
                !xDeviceFlag -> onError(RestoreUserSessionDisabledException())
                variant == UsercentricsVariant.CCPA -> onError(RestoreUserSessionNotSupportedException(variant.name))
                activeControllerId == controllerId -> onSuccessCallback()
                else -> doRestoreUserSession(controllerId, onSuccessCallback, onError)
            }
        }
    }

    private fun doRestoreUserSession(controllerId: String, onSuccess: (() -> Unit), onError: (UsercentricsException) -> Unit) {
        invokeClearUserSession()

        val selfHostedEnabled = options.isSelfHostedConfigurationValid()
        if (selfHostedEnabled) {
            // ignore billing for self hosted configurations
        } else {
            application.billingSessionLifecycleCallback.invoke()
        }

        application.dataFacadeInstance.restoreUserSession(
            controllerId,
            application.initialValuesStrategy.value.variant,
            onSuccess = {
                activeControllerId = controllerId
                getConsentsTriggeringMediationAndConsentsUpdateEvent()
                onSuccess()
            },
            onError = onError
        )
    }

    override fun getUserSessionData(): String {
        // FIXME: decide a good approach to do this

        val result = kotlin.runCatching {
            val storageInstance = application.storageInstance.value
            val sessionData = UserSessionData(
                consents = storageInstance.getUserSessionDataConsents(),
                controllerId = getControllerId(),
                language = storageInstance.getSettingsLanguage(),
                tcf = if (isTCFEnabled) {
                    val tcfData = storageInstance.getTCFData()
                    UserSessionDataTCF(
                        tcString = tcfData.tcString,
                        // TODO check if this model has changed
                        vendorsDisclosed = tcfData.vendorsDisclosedMap.keys.toList(),
                        acString = getAdditionalConsentModeData().acString
                    )
                } else {
                    // TODO check if this model has changed
                    null
                },
                ccpa = if (isCCPAEnabled()) {
                    UserSessionDataCCPA(
                        ccpaString = application.ccpaInstance.value.getCCPADataAsString(),
                        timestampInMillis = storageInstance.getCcpaTimestampInMillis() ?: 0L
                    )
                } else {
                    null
                }
            )
            application.jsonParserInstance.encodeToString(UserSessionData.serializer(), sessionData)
        }
        return result.getOrNull() ?: ""
    }

    override fun getUSPData(): CCPAData {
        return application.ccpaInstance.value.getCCPAData()
    }

    override fun setCMPId(id: Int) {
        if (isTCFEnabled) {
            tcfInstance.setCmpId(id)
        } else {
            application.logger.error(setCmpIdError)
        }
    }

    override fun getTCFData(callback: (TCFData) -> Unit) {
        application.dispatcher.dispatch {
            tcfInstance.getTCFData()
        }.onSuccess {
            application.dispatcher.dispatchMain {
                callback(it)
            }
        }
    }

    override fun changeLanguage(language: String, onSuccess: () -> Unit, onFailure: (UsercentricsError) -> Unit) {
        val settingsOrchestrator = application.settingsOrchestrator.value

        if (settingsOrchestrator.isLanguageAlreadySelected(language)) {
            onSuccess()
            return
        }

        if (!settingsOrchestrator.isLanguageAvailable(language)) {
            onFailure(LanguageNotAvailableException(language).asError())
            return
        }

        // TODO: verify this behaviour
        application.dispatcher.dispatch {
            val resultSettingsWithNewLanguage = settingsOrchestrator.loadSettings(controllerId = activeControllerId, language = language)
            val exception = resultSettingsWithNewLanguage.exceptionOrNull()
            if (exception != null) {
                throw exception
            }
            finishChangeLanguage(language)
        }.onSuccess {
            application.dispatcher.dispatchMain {
                onSuccess()
            }
        }.onFailure {
            application.dispatcher.dispatchMain {
                onFailure(UsercentricsError(cause = UsercentricsException(message = "", cause = it)))
            }
        }
    }

    private suspend fun finishChangeLanguage(language: String): Result<Unit> {
        val storageSettings = application.dataFacadeInstance.getMergedServicesAndSettingsFromStorage()
        val mergedSettings = storageSettings.mergedSettings
        val mergedServices = storageSettings.mergedServices

        application.settingsInstance.value.setSettings(
            mergedSettings.copy(services = application.settingsInstance.value.getSettings().services.updateServices(mergedServices))
        )

        application.storageInstance.value.saveSettings(mergedSettings, mergedServices)

        if (!isTCFEnabled) {
            return Result.success(Unit)
        }

        val changeLanguageResult = tcfInstance.changeLanguage(language = language)

        val changeLanguageException = changeLanguageResult.exceptionOrNull()
        if (changeLanguageException != null) {
            return Result.failure(changeLanguageException)
        }
        return Result.success(Unit)
    }

    override fun acceptAllForTCF(fromLayer: TCFDecisionUILayer, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        if (isTCFEnabled) {
            if (application.settingsInstance.value.isAdditionalConsentModeEnabled()) {
                application.additionalConsentModeService.value.acceptAll()
            }
            tcfInstance.acceptAllDisclosed(fromLayer)
        } else {
            application.logger.error(customUITCFError("acceptAllForTCF"))
        }
        return acceptAll(consentType)
    }

    override fun acceptAll(consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        val updatedServices = application.settingsInstance.value.getSettings().services.map {
            it.copy(consent = LegacyConsent(status = true, history = it.consent.history))
        }

        application.dataFacadeInstance.execute(
            activeControllerId,
            services = updatedServices,
            consentAction = UsercentricsConsentAction.ACCEPT_ALL_SERVICES,
            consentType = consentType
        )
        return getConsentsTriggeringMediationAndConsentsUpdateEvent()
    }

    override fun denyAllForTCF(fromLayer: TCFDecisionUILayer, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        if (isTCFEnabled) {
            if (application.settingsInstance.value.isAdditionalConsentModeEnabled()) {
                application.additionalConsentModeService.value.denyAll()
            }
            tcfInstance.denyAllDisclosed(fromLayer)
        } else {
            application.logger.error(customUITCFError("denyAllForTCF"))
        }
        return denyAll(consentType)
    }

    override fun denyAll(consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        val updatedServices = application.settingsInstance.value.getSettings().services.map {
            it.copy(consent = LegacyConsent(status = it.isEssential, history = it.consent.history))
        }

        application.dataFacadeInstance.execute(
            activeControllerId,
            services = updatedServices,
            consentAction = UsercentricsConsentAction.DENY_ALL_SERVICES,
            consentType = consentType
        )
        return getConsentsTriggeringMediationAndConsentsUpdateEvent()
    }

    override fun saveDecisionsForTCF(
        tcfDecisions: TCFUserDecisions,
        fromLayer: TCFDecisionUILayer,
        serviceDecisions: List<UserDecision>,
        consentType: UsercentricsConsentType
    ): List<UsercentricsServiceConsent> {

        if (isTCFEnabled) {
            if (application.settingsInstance.value.isAdditionalConsentModeEnabled()) {
                saveAdTechProvidersDecisions(tcfDecisions.adTechProviders)
            }
            tcfInstance.updateChoices(tcfDecisions, fromLayer)
        } else {
            application.logger.error(customUITCFError("saveDecisionsForTCF"))
        }
        return saveDecisions(serviceDecisions, consentType)
    }

    private fun saveAdTechProvidersDecisions(adTechProviders: List<AdTechProviderDecision>) {
        val consentedIds = adTechProviders.mapNotNull {
            if (!it.consent) {
                return@mapNotNull null
            }
            it.id
        }
        application.additionalConsentModeService.value.save(consentedIds)
    }

    override fun saveDecisions(decisions: List<UserDecision>, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        val allServices = application.settingsInstance.value.getSettings().services

        var gdprDecisions = decisions
        val hideNonIabOnFirstLayer = tcfInstance.getHideNonIabOnFirstLayer()
        if (isTCFEnabled && gdprDecisions.isEmpty() && hideNonIabOnFirstLayer) {
            gdprDecisions = generateGDPRDecisions(allServices)
        }

        val decisionsMap = gdprDecisions.associate { it.serviceId to it.consent }
        val updatedServices = allServices.filter { decisionsMap.containsKey(it.id) }.map {
            val newStatus = it.isEssential || (decisionsMap[it.id] ?: it.consent.status)
            it.copy(
                consent = LegacyConsent(
                    history = it.consent.history,
                    status = newStatus
                )
            )
        }

        if (updatedServices.isNotEmpty()) {
            application.dataFacadeInstance.execute(
                activeControllerId,
                services = updatedServices,
                consentAction = UsercentricsConsentAction.UPDATE_SERVICES,
                consentType = consentType
            )
        }
        return getConsentsTriggeringMediationAndConsentsUpdateEvent()
    }

    private fun generateGDPRDecisions(allServices: List<LegacyService>): List<UserDecision> {
        val serviceConsent = !tcfInstance.getGdprAppliesOnTCF()
        return allServices.map {
            UserDecision(serviceId = it.id, consent = serviceConsent)
        }
    }

    override fun saveOptOutForCCPA(isOptedOut: Boolean, consentType: UsercentricsConsentType): List<UsercentricsServiceConsent> {
        if (!isCCPAEnabled()) {
            application.logger.error(CCPAErrors.SETTINGS_UNDEFINED)

            return if (isOptedOut) {
                denyAll(consentType)
            } else {
                acceptAll(consentType)
            }
        }

        application.ccpaInstance.value.setCcpaStorage(isOptedOut)

        val consentAction = if (isOptedOut) UsercentricsConsentAction.DENY_ALL_SERVICES else UsercentricsConsentAction.ACCEPT_ALL_SERVICES

        val updatedServices = application.settingsInstance.value.getSettings().services.map {
            val status = if (it.isEssential) true else !isOptedOut
            it.copy(consent = LegacyConsent(status = status, history = it.consent.history))
        }

        application.dataFacadeInstance.execute(
            controllerId = activeControllerId,
            services = updatedServices,
            consentAction = consentAction,
            consentType = consentType,
        )

        return getConsentsTriggeringMediationAndConsentsUpdateEvent()
    }

    // FIXME: refactor this, method being used only for TV
    override fun getUIApplication(predefinedUIVariant: PredefinedUIVariant): PredefinedUIApplicationManager {
        val variant = application.initialValuesStrategy.value.variant
            ?: throw UsercentricsException("Usercentrics is still initializing. Please, check if you are trying to show the UI before the `isReady` was invoked.")

        application.predefinedUIMediator.storeVariant(predefinedUIVariant)
        track(UsercentricsAnalyticsEventType.CMP_SHOWN)

        return PredefinedUIApplicationManager(
            consentManager = PredefinedUIConsentManagerImpl(
                usercentricsSDK = this,
                variant = variant,
                controllerId = getControllerId(),
            ),
            logger = application.logger,
            cookieInformationService = application.cookieInformationService,
            bannerViewDataService = BannerViewDataServiceImpl(
                settingsService = application.settingsService,
                settingsLegacy = application.settingsInstance.value,
                translationService = application.translationService,
                tcfInstance = tcfInstance,
                ccpaInstance = application.ccpaInstance.value,
                additionalConsentModeService = application.additionalConsentModeService.value,
                variant = variant,
                dispatcher = application.dispatcher,
            ),
        )
    }

    override fun getUIFactoryHolder(
        abTestingVariant: String?,
        predefinedUIVariant: PredefinedUIVariant?,
        callback: (PredefinedUIFactoryHolder) -> Unit
    ) {
        assertUIThread()
        val legalVariant = application.initialValuesStrategy.value.variant
            ?: throw UsercentricsException("Usercentrics is still initializing. Please, check if you are trying to show the UI before the `isReady` was invoked.")

        abTestingVariant?.let { setABTestingVariant(it) }

        UsercentricsView(
            usercentricsSDK = this,
            variant = legalVariant,
            controllerId = getControllerId(),
            settingsService = application.settingsService,
            translationService = application.translationService,
            ccpaInstance = application.ccpaInstance.value,
            settingsLegacy = application.settingsInstance.value,
            tcfInstance = tcfInstance,
            additionalConsentModeService = application.additionalConsentModeService.value,
            dispatcher = application.dispatcher,
            logger = application.logger,
        ).getUIHolder { uiHolder ->
            storeVariant(predefinedUIVariant, uiHolder.data.settings)
            callback(PredefinedUIFactoryHolder(uiHolder, application.uiDependencyManager))
        }

        track(UsercentricsAnalyticsEventType.CMP_SHOWN)
    }

    override fun track(event: UsercentricsAnalyticsEventType) {
        // TODO should we use `application.settingsOrchestrator.value.settingsIdObservable` ?
        val settingsId = application.settingsOrchestrator.value.activeSettingsId
        application.analyticsFacade.value.report(eventType = event, settingsId = settingsId, abTestingVariant = getABTestingVariant())
    }

    override fun setABTestingVariant(variantName: String) {
        if (variantName.isBlank() || variantName == getABTestingVariant()) {
            // Nothing to select and avoid confusing logs
            return
        }

        val variantsSettings = application.settingsService.settings?.data?.variants

        val isEnabled = variantsSettings?.enabled == true
        assert(isEnabled) { "AB Testing is not enabled in the Admin Interface" }

        val settingsVariants = variantsSettings?.decodeVariants(application.jsonParserInstance) ?: emptyList()

        application.logger.debug("Select AB Testing Variant '$variantName'. Admin Interface list: $settingsVariants.")
        assert(settingsVariants.contains(variantName)) { "Selected AB Testing Variant '$variantName' is not configured in the Admin Interface list: $settingsVariants" }

        application.storageInstance.value.saveABTestingVariant(variantName)
    }

    override fun getABTestingVariant(): String? {
        return application.storageInstance.value.getABTestingVariant()
    }

    override fun getAdditionalConsentModeData(): AdditionalConsentModeData {
        return application.additionalConsentModeService.value.getData()
    }

    private fun setupABTestingIfNeeded() {
        val currentSelectedVariant = getABTestingVariant()
        if (!currentSelectedVariant.isNullOrBlank()) {
            application.logger.debug("AB Testing Variant was already selected '$currentSelectedVariant'.")
            return
        }

        val variantsSettings = application.settingsService.settings?.data?.variants
        val isEnabled = variantsSettings?.enabled == true
        val internalSelection = variantsSettings?.activateWith == VariantsSettings.activateWithUC
        if (isEnabled && internalSelection) {
            application.logger.debug("AB Testing 'Activate with Usercentrics' option triggered the variant selection.")
            val settingsVariants = variantsSettings?.decodeVariants(application.jsonParserInstance) ?: emptyList()
            val selected = settingsVariants.shuffled().firstOrNull() ?: ""
            setABTestingVariant(selected)
        }
    }

    override fun readyStatus(): UsercentricsReadyStatus {
        val settingsOrchestrator = application.settingsOrchestrator.value

        val geolocationRuleset = if (options.ruleSetId.isNotBlank()) {
            GeolocationRuleset(
                activeSettingsId = settingsOrchestrator.activeSettingsId,
                bannerRequiredAtLocation = settingsOrchestrator.noShow.not(),
            )
        } else {
            null
        }

        return UsercentricsReadyStatus(
            shouldCollectConsent = shouldCollectConsent(),
            consents = getConsents(),
            geolocationRuleset = geolocationRuleset,
            location = application.locationService.value.location,
        )
    }

    override fun clearUserSession(onSuccess: (UsercentricsReadyStatus) -> Unit, onError: (UsercentricsError) -> Unit) {
        application.dispatcher.dispatch {
            assertNotUIThread()
            invokeClearUserSession()
        }.onSuccess {
            application.dispatcher.dispatchMain {
                application.logger.debug("Clear User Session finished with success")

                onSuccess(readyStatus())
            }
        }.onFailure {
            application.dispatcher.dispatchMain {
                val error = UsercentricsException("Clear User Session failed", it).asError()
                application.logger.error(error)

                onError(error)
            }
        }
    }

    private fun invokeClearUserSession() {
        application.logger.debug("Clearing User Session")

        activeControllerId = ""

        application.storageInstance.value.clear()
        application.settingsInstance.value.clearConsents()

        if (isTCFEnabled) {
            // FIXME: TCF vendors are cleared --> potentially increasing the probability of ConcurrentModificationException
            application.tcfInstance.value.clearTCFConsentsData()
            if (application.settingsInstance.value.isAdditionalConsentModeEnabled()) {
                application.additionalConsentModeService.value.reset()
            }
        }

        application.initialValuesStrategy.value.loadConsents(isFirstInitialization = true, controllerId = "")


        getConsentsTriggeringMediationAndConsentsUpdateEvent()
    }

    private fun logConsentMediationInitialState() {
        if (options.consentMediation) {
            // FIXME: avoid using the entire object, map to only consents
            application.mediationFacade.value.logInitialState(getServices())
        }
    }

    // FIXME: change me to run only on the very first SDK initialization
    private fun runMediationAfterInitialize() {
        if (!options.consentMediation) {
            return
        }

        val consentsList = getConsents()

        // FIXME: dupe of the method below
        if (isTCFEnabled) {
            getTCFData { tcfData ->
                applyMediationIfNeeded(consentsList, mapTCFConsentPayload(tcfData))
            }
        } else {
            applyMediationIfNeeded(consents = consentsList, tcfConsentPayload = null)
        }
    }

    private fun getConsentsTriggeringMediationAndConsentsUpdateEvent(): List<UsercentricsServiceConsent> {
        val consentsList = getConsents()

        if (!isTCFEnabled) {
            applyMediationIfNeeded(consents = consentsList, tcfConsentPayload = null)
            emitUpdatedConsentEvent(consentsList = consentsList)
            return consentsList
        }

        getTCFData { tcfData ->
            applyMediationIfNeeded(consentsList, mapTCFConsentPayload(tcfData))

            emitUpdatedConsentEvent(
                consentsList = consentsList,
                tcString = tcfData.tcString,
                acString = getAdditionalConsentModeData().acString,
            )
        }
        return consentsList
    }

    private fun mapTCFConsentPayload(tcfData: TCFData): TCFConsentPayload {
        return TCFConsentPayload(
            eea = application.locationService.value.location.isInEU(),
            purposes = tcfData.purposes,
            vendors = tcfData.vendors,
        )
    }

    private fun applyMediationIfNeeded(consents: List<UsercentricsServiceConsent>, tcfConsentPayload: TCFConsentPayload?) {
        if (!options.consentMediation) {
            return
        }

        application.dispatcher.dispatch {
            val ccpaOptedOut = if (isCCPAEnabled()) {
                getUSPData().optedOut ?: false
            } else {
                null
            }

            val mediationPayload = ConsentMediationPayload(
                dps = consents.associate { it.templateId to it.status },
                tcf = tcfConsentPayload,
                ccpaOptedOut = ccpaOptedOut,
                variant = application.initialValuesStrategy.value.variant!!,
            )
            application.mediationFacade.value.mediateConsents(mediationPayload)
        }.onSuccess {
            application.dispatcher.dispatchMain { UsercentricsEvent.mediationConsentEvent.emit(it) }
        }
    }

    private fun emitUpdatedConsentEvent(consentsList: List<UsercentricsServiceConsent>, tcString: String = "", acString: String = "") {
        val value = UpdatedConsentPayload(
            consents = consentsList,
            controllerId = getControllerId(),
            tcString = tcString,
            uspString = getUSPStringIfAvailable(),
            acString = acString,
        )
        application.dispatcher.dispatchMain { UsercentricsEvent.updatedConsentEvent.emit(value) }
    }

    private fun getUSPStringIfAvailable(): String {
        if (isCCPAEnabled()) {
            return getUSPData().uspString
        }
        return ""
    }

    private fun isCCPAEnabled(): Boolean {
        return application.settingsInstance.value.isCCPAEnabled()
    }

    private fun storeVariant(predefinedUIVariant: PredefinedUIVariant?, settings: PredefinedUIViewSettings) {
        val variant = predefinedUIVariant ?: settings.firstLayerV2.layout.toPredefinedUIVariant()
        application.predefinedUIMediator.storeVariant(variant)
    }
}

