package com.usercentrics.sdk.core.settings

import com.usercentrics.sdk.Observable
import com.usercentrics.sdk.UsercentricsOptions
import com.usercentrics.sdk.core.application.MainApplication
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.models.api.ApiConstants
import com.usercentrics.sdk.models.api.ApiErrors
import com.usercentrics.sdk.services.settings.ISettingsLegacy
import com.usercentrics.sdk.v2.location.data.LocationAwareResponse
import kotlinx.coroutines.job
import kotlin.coroutines.coroutineContext

internal class SettingsOrchestratorImpl(
    private val application: MainApplication
) : SettingsOrchestrator {

    private var jsonFileVersion = ""

    override val settingsIdObservable: Observable<String> = Observable()
    override var activeSettingsId = ""

    override var jsonFileLanguage = ""
    override var noShow = false

    private var allSettingsIds = emptySet<String>()

    /**
     * If languages etag changed, then we should request all necessary data again in order to avoid some cache control bugs
     * */
    private var languageEtagChanged: Boolean = false

    override suspend fun boot(options: UsercentricsOptions) {
        jsonFileLanguage = options.defaultLanguage
        jsonFileVersion = options.version.ifBlank { ApiConstants.FALLBACK_VERSION }

        val settingsId = options.settingsId
        if (settingsId.isNotBlank()) {
            setActiveSettingsId(active = settingsId, allSettingIds = setOf(settingsId))
            return
        }

        val sessionGeoRule = application.ruleSetService.value.getActiveSettingsId(options.ruleSetId)
        setActiveSettingsId(active = sessionGeoRule.activeSettingsId, allSettingIds = sessionGeoRule.allSettingsIds)
        this.noShow = sessionGeoRule.noShow

        application.locationService.value.set(sessionGeoRule.location)
    }

    @Suppress("FoldInitializerAndIfToElvis")
    override suspend fun coldInitialize(controllerId: String): Result<Unit> {
        deleteDeprecatedSettingsIds()

        val locationService = application.locationService.value
        val hasCachedLocation = locationService.loadLocation()

        val result = application.languageFacade.value.resolveLanguage(
            settingsId = activeSettingsId,
            version = jsonFileVersion,
            defaultLanguage = jsonFileLanguage,
            bypassCache = !hasCachedLocation
        )

        val locationAwareResponse = result.getOrNull()
        if (locationAwareResponse == null) {
            return Result.failure(result.exceptionOrNull() ?: UsercentricsException(ApiErrors.FETCH_AVAILABLE_LANGUAGES))
        }

        languageEtagChanged = locationAwareResponse.languageEtagChanged

        val resultLoadSettings = loadSettingsFromAdmin(controllerId = controllerId, locationAwareResponse = locationAwareResponse)

        val exceptionWhenLoadingSettings = resultLoadSettings.exceptionOrNull()
        if (exceptionWhenLoadingSettings != null) {
            return Result.failure(exceptionWhenLoadingSettings)
        }
        return Result.success(Unit)
    }

    private fun deleteDeprecatedSettingsIds() {
        application.storageInstance.value.deleteSettingsThatDoNotMatch(allSettingsIds)
    }

    override suspend fun loadSettings(controllerId: String, language: String?): Result<Unit> {
        val settingsId = activeSettingsId

        val settingsInitializationParameters = SettingsInitializationParameters(
            settingsId = settingsId,
            jsonFileVersion = jsonFileVersion,
            jsonFileLanguage = language ?: jsonFileLanguage,
            controllerId = controllerId,
            languageEtagChanged = languageEtagChanged
        )
        val result = application.settingsInstance.value.initSettings(settingsInitializationParameters)

        val exceptionWhenLoadingSettings = result.exceptionOrNull()
        if (exceptionWhenLoadingSettings != null) {
            return Result.failure(exceptionWhenLoadingSettings)
        }

        if (language != null) {
            jsonFileLanguage = language
        }
        return result
    }

    override fun isLanguageAlreadySelected(language: String): Boolean {
        return language == jsonFileLanguage
    }

    override fun isLanguageAvailable(language: String): Boolean {
        val baseSettings = application.settingsInstance.value.getSettings()
        val availableLanguages = when {
            baseSettings.ui != null -> baseSettings.ui.language.available.map { it.isoCode }
            baseSettings.tcfui != null -> baseSettings.tcfui.language.available.map { it.isoCode }
            else -> emptyList()
        }
        return language in availableLanguages
    }

    private suspend fun loadSettingsFromAdmin(controllerId: String, locationAwareResponse: LocationAwareResponse<String>): Result<Unit> {
        updateLocationServiceIfNeeded(locationAwareResponse)

        val selectedLanguage = locationAwareResponse.data

        jsonFileLanguage = selectedLanguage
        application.logger.debug("Language: $selectedLanguage")

        val loadSettingsResult = loadSettings(controllerId = controllerId)

        val exceptionWhenLoadingSettings = loadSettingsResult.exceptionOrNull()
        if (exceptionWhenLoadingSettings != null) {
            return Result.failure(exceptionWhenLoadingSettings)
        }

        val finalInitResult = initSettingsCallback(controllerId = controllerId)

        val exceptionOnFinalInit = finalInitResult.exceptionOrNull()
        if (exceptionOnFinalInit != null) {
            return Result.failure(exceptionOnFinalInit)
        }
        return Result.success(Unit)
    }

    private fun updateLocationServiceIfNeeded(locationAwareResponse: LocationAwareResponse<String>) {
        val locationService = application.locationService.value

        locationService.loadLocation()
        if (!locationAwareResponse.location.isEmpty()) {
            locationService.set(locationAwareResponse.location)
        }
    }

    private suspend fun initSettingsCallback(controllerId: String): Result<Unit> {
        val settingsInstance = application.settingsInstance.value

        val tcfEnabled = settingsInstance.isTCFEnabled()
        try {
            if (tcfEnabled) {
                return initTCFAndAdditionalConsentMode(settingsInstance, controllerId)
            } else {
                wipeLocalStorageForNonTCFSettingsId()
            }
        } finally {
            if (coroutineContext.job.isCancelled && !tcfEnabled) {
                // in case this thread gets cancelled
                // make sure to execute wipe before moving on
                wipeLocalStorageForNonTCFSettingsId()
            }
        }

        if (settingsInstance.isCCPAEnabled()) {
            application.ccpaInstance.value.initialize(settingsInstance.getCCPAIABAgreementExists())
        }
        return finishInitialization(controllerId)
    }

    private fun wipeLocalStorageForNonTCFSettingsId() {
        application.storageInstance.value.apply {
            saveActualTCFSettingsId("")
            clearTCFStorageEntries()
        }
    }

    private suspend fun initTCFAndAdditionalConsentMode(settingsInstance: ISettingsLegacy, controllerId: String): Result<Unit> {
        val tcfInitializeResult = application.tcfInstance.value.initialize(settingsId = activeSettingsId)

        val exceptionWhenInitializingTCF = tcfInitializeResult.exceptionOrNull()
        if (exceptionWhenInitializingTCF != null) {
            return Result.failure(exceptionWhenInitializingTCF)
        }

        if (settingsInstance.isAdditionalConsentModeEnabled()) {
            val additionalConsentModeResult = initAdditionalConsentMode()
            val exceptionAdditionalConsentMode = additionalConsentModeResult.exceptionOrNull()

            if (exceptionAdditionalConsentMode != null) {
                return Result.failure(exceptionAdditionalConsentMode)
            }
        }
        return finishInitialization(controllerId)
    }

    private suspend fun initAdditionalConsentMode(): Result<Unit> {
        return try {
            val selectedAdTechProviders = application.settingsInstance.value.selectedAdTechProviders()
            application.additionalConsentModeService.value.load(selectedAdTechProviders)
            Result.success(Unit)
        } catch (ex: Exception) {
            Result.failure(UsercentricsException(ApiErrors.FETCH_TCF_DATA, ex))
        }
    }

    private suspend fun finishInitialization(controllerId: String): Result<Unit> {
        val isFirstInitialization = controllerId.isBlank()

        return try {
            application.initialValuesStrategy.value.boot(isFirstInitialization, controllerId)
            checkValidState()

            Result.success(Unit)
        } catch (ex: Exception) {
            Result.failure(UsercentricsException("There was a failure during the initialization", ex))
        } finally {
            if (coroutineContext.job.isCancelled && isFirstInitialization) {
                wipeStorage()
            }
        }
    }

    private fun wipeStorage() {
        application.logger.debug("Storage wiped out, given failed initialization and 1st interaction with SDK")
        application.storageInstance.value.clear()
    }

    private fun checkValidState() {
        val location = application.locationService.value.location
        if (location.isEmpty()) {
            throw IllegalStateException("Location cannot be empty")
        }
        application.initialValuesStrategy.value.variant ?: throw IllegalStateException("No variant value")
    }

    private fun setActiveSettingsId(active: String, allSettingIds: Set<String>) {
        this.activeSettingsId = active
        this.allSettingsIds = allSettingIds

        application.storageInstance.value.bootSettings(activeSettingsId)

        this.settingsIdObservable.emit(active)
    }
}
