package com.usercentrics.sdk.domain.api.http

import com.usercentrics.sdk.assertNotUIThread
import com.usercentrics.sdk.errors.UsercentricsTimeoutException
import com.usercentrics.sdk.ui.userAgent.UserAgentProvider
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import kotlinx.coroutines.InternalCoroutinesApi
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

internal class HttpRequestsImpl(
    private val httpClient: HttpClient,
    private val userAgentProvider: UserAgentProvider,
    private val disptacher: Dispatcher,
) : HttpRequests {

    override fun get(url: String, headers: Map<String, String>?, onSuccess: (HttpResponse) -> Unit, onError: (Throwable) -> Unit) {
        disptacher.dispatch {
            getSync(url, headers)
        }.onSuccess {
            onSuccess(it)
        }.onFailure {
            onError(it)
        }
    }

    override fun post(url: String, bodyData: String, headers: Map<String, String>?, onSuccess: (String) -> Unit, onError: (Throwable) -> Unit) {
        disptacher.dispatch {
            postSync(url, bodyData, headers)
        }.onSuccess {
            onSuccess(it)
        }.onFailure {
            onError(it)
        }
    }

    // used by requests on demand (cookie information, billing and getConsents)
    override fun getSync(url: String, headers: Map<String, String>?): HttpResponse {
        assertNotUIThread()
        return httpClient.get(url, appendUserAgent(headers))
    }

    @OptIn(InternalCoroutinesApi::class)
    override suspend fun getSync2(url: String, headers: Map<String, String>?): HttpResponse {
        assertNotUIThread()

        return coroutineScope {
            suspendCancellableCoroutine { continuation ->
                val onSuccess: (HttpResponse) -> Unit = {
                    continuation.resume(it)
                }

                val onError: (Throwable) -> Unit = {
                    continuation.resumeWithException(it)
                }

                val disposable = httpClient.get(url = url, headers = appendUserAgent(headers), onSuccess = onSuccess, onError = onError)

                continuation.invokeOnCancellation {
                    disposable.disconnect()
                    continuation.tryResumeWithException(UsercentricsTimeoutException())
                }
            }
        }
    }

    override fun postSync(url: String, bodyData: String, headers: Map<String, String>?): String {
        assertNotUIThread()
        return httpClient.post(url, appendUserAgent(headers), bodyData)
    }

    private fun appendUserAgent(headers: Map<String, String>?): Map<String, String> {
        val encodedUserAgent = userAgentProvider.provide().encode()
        val result = mutableMapOf("User-Agent" to encodedUserAgent)
        headers?.forEach {
            result[it.key] = it.value
        }
        return result
    }
}
