package com.usercentrics.sdk.mediation.data

import com.usercentrics.sdk.services.tcf.interfaces.TCFPurpose
import com.usercentrics.sdk.services.tcf.interfaces.TCFVendor

internal data class TCFConsentPayload(
    val eea: Boolean,
    val purposes: Map<Int, TCFConsentValue>,
    val vendors: Map<Int, TCFConsentValue>,
) {

    constructor(eea: Boolean, purposes: List<TCFPurpose>, vendors: List<TCFVendor>) : this(
        eea = eea,
        purposes = purposes.associate { purpose ->
            purpose.id to TCFConsentValue(
                consent = valueAwareOfToggleVisibility(purpose.consent, purpose.showConsentToggle),
                legitimateInterest = valueAwareOfToggleVisibility(purpose.legitimateInterestConsent, purpose.showLegitimateInterestToggle)
            )
        },
        vendors = vendors.associate { vendor ->
            vendor.id to TCFConsentValue(
                consent = valueAwareOfToggleVisibility(vendor.consent, vendor.showConsentToggle),
                legitimateInterest = valueAwareOfToggleVisibility(vendor.legitimateInterestConsent, vendor.showLegitimateInterestToggle)
            )
        }
    )

    fun buildGranularConsent(vendorId: Int): MediationGranularConsent? {
        val vendorConsentValue = vendors[vendorId] ?: return null

        // reference: https://developers.google.com/tag-platform/security/guides/implement-TCF-strings#tag_behavior

        return MediationGranularConsent(
            eea = this.eea,
            analyticsStorage = consentForPurposes(vendorConsentValue, 9, 10),
            adStorage = consentForPurposes(vendorConsentValue, 1),
            adUserData = consentForPurposes(vendorConsentValue, 1, 7),
            adPersonalization = consentForPurposes(vendorConsentValue, 3, 4),
        )
    }

    private fun consentForPurposes(vendorConsentValue: TCFConsentValue, vararg purposeIds: Int): Boolean {
        return purposeIds.all { purposeId ->
            val purposeConsentValue = this.purposes[purposeId] ?: return@all false

            // considering legitimate interest as a fallback in case there's no consent set
            // only purpose can be used with legitimate interest legal basis
            if (purposeId !in setOf(1, 3, 4, 5, 6)) {
                if (purposeConsentValue.consent == false && vendorConsentValue.consent == false) {
                    if (purposeConsentValue.legitimateInterest != null) {
                        if (vendorConsentValue.legitimateInterest != null) {
                            return@all vendorConsentValue.legitimateInterest && purposeConsentValue.legitimateInterest
                        }
                        return purposeConsentValue.legitimateInterest
                    }
                }
            }

            if (purposeConsentValue.consent != null) {
                if (vendorConsentValue.consent != null) {
                    return@all vendorConsentValue.consent && purposeConsentValue.consent
                }
                return@all purposeConsentValue.consent
            }

            if (purposeConsentValue.legitimateInterest != null) {
                if (vendorConsentValue.legitimateInterest != null) {
                    return@all vendorConsentValue.legitimateInterest && purposeConsentValue.legitimateInterest
                }
                return purposeConsentValue.legitimateInterest
            }
            return@all false
        }
    }

    companion object {

        private fun valueAwareOfToggleVisibility(consent: Boolean?, showToggle: Boolean): Boolean? {
            if (showToggle) {
                return consent
            }
            return null
        }
    }
}
