package com.usercentrics.sdk.mediation.facade

import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.mediation.data.*
import com.usercentrics.sdk.mediation.data.ConsentApplyResult
import com.usercentrics.sdk.mediation.data.ConsentMediationPayload
import com.usercentrics.sdk.mediation.service.IMediationService
import com.usercentrics.sdk.v2.settings.data.UsercentricsService

internal class MediationFacade(
    private val mediationService: IMediationService,
    private val logger: UsercentricsLogger,
) : IMediationFacade {

    companion object {
        private const val MEDIATION_TAG = "[Mediation]"
    }

    override fun logInitialState(services: List<UsercentricsService>) {
        logger.debug("Consent Mediation is Enabled")

        val servicesNamesSupported = mutableListOf<String>()
        services.forEach { service ->
            val templateId = service.templateId ?: return@forEach

            if (mediationService.isSDKSupported(templateId)) {
                servicesNamesSupported.add(service.dataProcessor ?: "")
            }
        }
        logger.debug("$MEDIATION_TAG ${servicesNamesSupported.size}/${services.size} Services are supported: ${servicesNamesSupported.joinToString(separator = " | ")}")
    }

    override fun mediateConsents(consentMediationPayload: ConsentMediationPayload): MediationResultPayload {
        val result = mediationService.applyConsents(consentMediationPayload)

        result.applied.forEach {
            val message = mediationLogMessage(it)
            logger.debug(message)
        }

        // FIXME: improve this mapping to contain granular consents
        return MediationResultPayload(applied = result.applied.map {
            ConsentApplied(
                name = it.name,
                templateId = it.templateId ?: "",
                consent = it.consent ?: false,
                mediated = it.mediated,
            )
        })
    }

    private fun mediationLogMessage(applyResult: ConsentApplyResult): String {
        val hasGranularConsent = applyResult.granularConsent != null

        if (!applyResult.mediated) {
            return "$MEDIATION_TAG ${applyResult.name} - Unable to pass${if (hasGranularConsent) "Granular" else ""} consent. " +
                    "Please, report this issue to Usercentrics. https://usercentricssupport.zendesk.com/hc/en-us/requests/new"
        }

        if (hasGranularConsent) {
            return "$MEDIATION_TAG Applied Granular Consent to ${applyResult.name} - ${applyResult.granularConsent}"
        }

        return "$MEDIATION_TAG Applied ${applyResult.name} - Consent is ${applyResult.consent.toString().uppercase()}"
    }
}
