package com.usercentrics.sdk.mediation.service

import com.usercentrics.sdk.mediation.MediationSDK
import com.usercentrics.sdk.mediation.data.ConsentApplyResult
import com.usercentrics.sdk.mediation.data.ConsentMediationPayload
import com.usercentrics.sdk.mediation.data.MediationResultPayloadDTO
import com.usercentrics.sdk.mediation.data.TCFConsentPayload
import com.usercentrics.sdk.mediation.sdk.AdjustMediation

internal class MediationService(
    private val sdks: Map<String, MediationSDK>,
    private val granularSDKs: List<MediationSDK> = listOf(),
    private val adjust: AdjustMediation,
) : IMediationService {

    override fun isSDKSupported(templateId: String): Boolean {
        // FIXME: enhance this to also log Consent Mode for Google when there's Google Ad Vendor present
        return sdks.containsKey(templateId) || adjust.templateId == templateId
    }

    override fun applyConsents(payload: ConsentMediationPayload): MediationResultPayloadDTO {
        val applied = mutableListOf<ConsentApplyResult>()

        val consentedTemplateIds = payload.dps.keys
        val isAdjustInDPSList = adjust.isAvailable(consentedTemplateIds)

        if (payload.tcf != null) {
            applied.addAll(mediateGranularConsents(isAdjustInDPSList, payload.tcf))
        }
        applied.addAll(mediateDPSConsents(isAdjustInDPSList, payload))

        return MediationResultPayloadDTO(applied)
    }

    private fun mediateGranularConsents(adjustInDPSList: Boolean, tcfPayload: TCFConsentPayload): List<ConsentApplyResult> {
        val applied = mutableListOf<ConsentApplyResult>()
        val vendorsAppliedInAdjust = mutableSetOf<Int>()

        granularSDKs.forEach { mediationSDK ->
            val vendorId = mediationSDK.vendorId ?: return@forEach
            val granularConsent = tcfPayload.buildGranularConsent(vendorId) ?: return@forEach

            // for TCF, Firebase SDKs get the TCString automatically and set their flags
            // but Google still requires to call the `Firebase.analytics.setAnalyticsCollectionEnabled`
            // so this routine is calling the API and logging for customers
            val mediated = mediationSDK.applyGranular(granularConsent)
            applied.add(ConsentApplyResult(name = "${mediationSDK.name} (Granular Signal)", mediated = mediated, granularConsent = granularConsent))

            if (adjustInDPSList && !vendorsAppliedInAdjust.contains(vendorId)) {
                val signaledToAdjust = adjust.signalGranularConsent(vendorId = vendorId, granularConsent = granularConsent)
                applied.add(ConsentApplyResult(name = "${mediationSDK.name} (Adjust Signal)", mediated = signaledToAdjust, granularConsent = granularConsent))

                vendorsAppliedInAdjust.add(vendorId)
            }
        }
        return applied
    }

    private fun mediateDPSConsents(adjustInDPSList: Boolean, payload: ConsentMediationPayload): List<ConsentApplyResult> {
        val results = mutableListOf<ConsentApplyResult>()

        payload.dps.forEach { dpsConsent ->
            val templateId = dpsConsent.key
            val consent = serviceConsentValue(payload.ccpaOptedOut, dpsConsent.value)

            if (adjustInDPSList && adjust.canMediate(templateId)) {
                val adjustSettings = adjust.apply(templateId, consent)
                results.add(ConsentApplyResult(name = adjustSettings.name, templateId = templateId, consent = consent, mediated = adjustSettings.mediated))
                return@forEach
            }

            val mediationSDK = sdks[templateId] ?: return@forEach

            val mediated = mediationSDK.apply(consent = consent, variant = payload.variant)
            results.add(ConsentApplyResult(name = mediationSDK.name, templateId = templateId, consent = consent, mediated = mediated))
        }
        return results
    }

    private fun serviceConsentValue(ccpaOptedOut: Boolean?, serviceConsent: Boolean): Boolean {
        if (ccpaOptedOut == null) {
            return serviceConsent
        }
        if (serviceConsent) {
            return true
        }
        return ccpaOptedOut.not()
    }
}
