package com.usercentrics.sdk.models.settings

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.extensions.sortedAlphaBy
import com.usercentrics.sdk.models.gdpr.DefaultUISettings
import com.usercentrics.sdk.models.tcf.TCFUISettings
import com.usercentrics.sdk.v2.settings.data.CCPASettings
import com.usercentrics.sdk.v2.settings.data.ConsentDisclosureObject
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory

internal data class LegacyExtendedSettings(
    val categories: List<UsercentricsCategory> = listOf(),
    val services: List<LegacyService> = listOf(),
    val gdpr: GDPROptions? = null,
    val ccpa: CCPASettings? = null,
    var controllerId: String = "",
    val id: String = "",
    val isTcfEnabled: Boolean = false,
    val showFirstLayerOnVersionChange: List<Int> = listOf(),
    val tcfui: TCFUISettings? = null,
    val ui: DefaultUISettings? = null,
    val version: String = "",
    val framework: String? = null,
    val restoredSessionLastInteractionTimestamp: Long? = null,
    val renewConsentsTimestampInSeconds: Long? = null,
)

internal enum class SettingsVersion {
    MAJOR,
    MINOR,
    PATCH
}

internal data class GDPROptions(
    val displayCmpOnlyToEUUsers: Boolean,
    val reshowCmpInMonths: Int? = 0,
)

internal data class LegacyService(
    val dataCollected: List<String>,
    val dataDistribution: PredefinedUIDataDistribution,
    val dataPurposes: List<String>,
    val dataRecipients: List<String>,
    val serviceDescription: String,
    val id: String,
    val legalBasis: List<String>,
    val name: String,
    val processingCompany: PredefinedUIProcessingCompany,
    val retentionPeriodDescription: String = "",
    val technologiesUsed: List<String>,
    val urls: PredefinedUIURLs,
    val version: String,
    val categorySlug: String,
    val categoryLabel: String,
    var consent: LegacyConsent,
    val isEssential: Boolean,
    val disableLegalBasis: Boolean,
    val processorId: String,
    val subServices: List<LegacyBasicService>,
    val cookieMaxAgeSeconds: Long? = null,
    val usesNonCookieAccess: Boolean?,
    val deviceStorageDisclosureUrl: String?,
    val deviceStorage: ConsentDisclosureObject?,
    val isHidden: Boolean,
    val defaultConsentStatus: Boolean? = null,
)

internal fun List<LegacyService>.sortByName(): List<LegacyService> {
    return this.sortedAlphaBy { it.name }
}

internal fun List<LegacyService>.updateServices(updates: List<LegacyService>): List<LegacyService> {
    val servicesMap = this.associateBy { it.id }.toMutableMap()
    updates.forEach {
        servicesMap[it.id] = it
    }
    return servicesMap.values.toList()
}

internal data class LegacyBasicService(
    val dataCollected: List<String>,
    val dataDistribution: PredefinedUIDataDistribution,
    val dataPurposes: List<String>,
    val dataRecipients: List<String>,
    val serviceDescription: String,
    val id: String,
    val legalBasis: List<String>,
    val name: String,
    val disableLegalBasis: Boolean?,
    val processingCompany: PredefinedUIProcessingCompany,
    val retentionPeriodDescription: String = "",
    val technologiesUsed: List<String>,
    val urls: PredefinedUIURLs,
    val version: String,
    val cookieMaxAgeSeconds: Long? = null,
    val usesNonCookieAccess: Boolean?,
    val deviceStorageDisclosureUrl: String?,
    val deviceStorage: ConsentDisclosureObject?,
    val dpsDisplayFormat: String? = null,
    val isHidden: Boolean
)

data class LegacyConsent(
    val history: List<LegacyConsentHistoryEntry>,
    val status: Boolean
)

data class LegacyConsentHistoryEntry(
    val action: UsercentricsConsentAction,
    val status: Boolean,
    val type: UsercentricsConsentType,
    val language: String,
    val timestampInMillis: Long,
) {
    val formattedDate: String
        get() = DateTime(timestampInMillis).formatLocalTimezone()
}

enum class UsercentricsConsentAction(internal val text: String) {
    ACCEPT_ALL_SERVICES("onAcceptAllServices"),
    DENY_ALL_SERVICES("onDenyAllServices"),
    ESSENTIAL_CHANGE("onEssentialChange"),
    INITIAL_PAGE_LOAD("onInitialPageLoad"),
    NON_EU_REGION("onNonEURegion"),
    SESSION_RESTORED("onSessionRestored"),
    TCF_STRING_CHANGE("onTcfStringChange"),
    UPDATE_SERVICES("onUpdateServices");

    companion object {
        internal fun from(s: String): UsercentricsConsentAction? = values().find { it.text.equals(s, ignoreCase = true) }
    }

    fun getType(): UsercentricsConsentType {
        return when (this) {
            ACCEPT_ALL_SERVICES -> UsercentricsConsentType.EXPLICIT
            DENY_ALL_SERVICES -> UsercentricsConsentType.EXPLICIT
            ESSENTIAL_CHANGE -> UsercentricsConsentType.IMPLICIT
            INITIAL_PAGE_LOAD -> UsercentricsConsentType.IMPLICIT
            NON_EU_REGION -> UsercentricsConsentType.IMPLICIT
            SESSION_RESTORED -> UsercentricsConsentType.IMPLICIT
            TCF_STRING_CHANGE -> UsercentricsConsentType.EXPLICIT
            UPDATE_SERVICES -> UsercentricsConsentType.EXPLICIT
        }
    }
}

