package com.usercentrics.sdk.models.settings

import com.usercentrics.sdk.AdTechProvider
import com.usercentrics.sdk.CategoryProps
import com.usercentrics.sdk.DeviceLanguage
import com.usercentrics.sdk.models.api.SERVICE_DEFAULT_DISABLE_LEGAL_BASIS
import com.usercentrics.sdk.services.tcf.interfaces.TCFFeature
import com.usercentrics.sdk.services.tcf.interfaces.TCFSpecialPurpose
import com.usercentrics.sdk.v2.settings.data.DpsDisplayFormat
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.FirstLayerMobileVariant

internal object PredefinedUIData {
    const val CONTENT_SEPARATOR = "\n\n"
    const val BULLET_POINT = '\u2022'
}

class PredefinedUIViewSettings(
    val customization: PredefinedUICustomization,
    val internationalizationLabels: PredefinedUILabels,
    val firstLayerV2: UCUIFirstLayerSettings,
    val secondLayerV2: UCUISecondLayerSettings,
)

class UCUIFirstLayerSettings(
    val layout: FirstLayerMobileVariant = defaultLayout,
    val headerSettings: PredefinedUIHeaderSettings,
    val footerSettings: PredefinedUIFooterSettings,
    val contentSettings: List<PredefinedUICardUISection>
) {
    companion object {
        internal val defaultLayout = FirstLayerMobileVariant.SHEET
    }
}

class UCUISecondLayerSettings(
    val headerSettings: PredefinedUIHeaderSettings,
    val footerSettings: PredefinedUIFooterSettings,
    val contentSettings: List<PredefinedUITabSettings>,
)

class PredefinedUITabSettings(
    val title: String,
    val content: PredefinedUITabContent
)

sealed interface PredefinedUITabContent {
    val cardUISections: List<PredefinedUICardUISection>
}

class PredefinedUIServicesContent(
    override val cardUISections: List<PredefinedUICardUISection>
) : PredefinedUITabContent

class PredefinedUICategoriesContent(
    override val cardUISections: List<PredefinedUICardUISection>
) : PredefinedUITabContent

interface PredefinedUIHeaderSettings{
    val title: String
    val contentDescription: String
    val links: List<PredefinedUILink>
    val logoPosition: FirstLayerLogoPosition
    val logoURL: String?
    val language: PredefinedUILanguageSettings?
    val firstLayerCloseLink: String?
    val firstLayerCloseIcon: Boolean?
}

class PredefinedUIFirstLayerHeaderSettings(
    override val title: String,
    val shortDescription: String?,
    override val contentDescription: String,
    override val links: List<PredefinedUILink>,
    override val logoPosition: FirstLayerLogoPosition,
    override val logoURL: String?,
    override val language: PredefinedUILanguageSettings?,
    override val firstLayerCloseLink: String?,
    override val firstLayerCloseIcon: Boolean?,
    val readMoreText: String,
) : PredefinedUIHeaderSettings

class PredefinedUISecondLayerHeaderSettings(
    override val title: String,
    override val contentDescription: String,
    override val links: List<PredefinedUILink>,
    override val logoPosition: FirstLayerLogoPosition,
    override val logoURL: String?,
    override val language: PredefinedUILanguageSettings?,
    override val firstLayerCloseLink: String?,
    override val firstLayerCloseIcon: Boolean?
) : PredefinedUIHeaderSettings

data class PredefinedUILanguageSettings(
    val available: List<PredefinedUILanguage>,
    val selected: PredefinedUILanguage
)

data class PredefinedUILanguage(
    val isoCode: String
) {
    val fullName: String = DeviceLanguage().getFullLanguageName(isoCode)
}

class PredefinedUIFooterSettings(
    val poweredBy: PredefinedUIFooterEntry? = null,
    val optOutToggle: PredefinedUIFooterEntry? = null,
    val optOutToggleInitialValue: Boolean = false,
    val buttons: List<List<PredefinedUIFooterButton>>,
    val buttonsLandscape: List<List<PredefinedUIFooterButton>>,
)

open class PredefinedUIFooterEntry(val label: String)

class PredefinedUIFooterButton(
    label: String,
    val type: PredefinedUIButtonType,
    val customization: PredefinedUICustomizationColorButton
) : PredefinedUIFooterEntry(label)

class PredefinedUILabels(
    val general: PredefinedUIGeneralLabels,
    val service: PredefinedUIServiceLabels,
    val firstLayerButtonLabels: FirstLayerButtonLabels,
    val cookieInformationLabels: PredefinedUICookieInformationLabels?,
    val ariaLabels: PredefinedUIAriaLabels,
)

data class PredefinedUICardUISection(
    val title: String?,
    val cards: List<PredefinedUICardUI> = emptyList(),
    val controllerID: PredefinedUIControllerIDSettings? = null
)

data class PredefinedUICardUI(
    val id: String,
    val title: String,
    val shortDescription: String?,
    val mainSwitchSettings: PredefinedUISwitchSettingsUI?,
    val content: PredefinedUICardContent?,
    val switchSettings: List<PredefinedUISwitchSettingsUI>? = null,
    val dependantSwitchSettings: List<PredefinedUIDependantSwitchSettings>? = null,
) {

    internal constructor(
        entry: CategoryProps,
        content: PredefinedUICardContent?,
        shortDescription: String?
    ) : this(
        entry = entry,
        mainSwitchSettings = PredefinedUISwitchSettingsUI(
            id = PredefinedUIDecision.CONSENT_ID,
            disabled = entry.category.isEssential,
            currentValue = entry.services.any { service -> service.consent.status },
        ),
        content = content,
        shortDescription = shortDescription,
        dependantSwitchSettings = entry.services.map { service ->
            PredefinedUIDependantSwitchSettings(
                service
            )
        },
    )

    internal constructor(
        entry: CategoryProps,
        mainSwitchSettings: PredefinedUISwitchSettingsUI?,
        content: PredefinedUICardContent?,
        shortDescription: String?,
        dependantSwitchSettings: List<PredefinedUIDependantSwitchSettings>? = null
    ) : this(
        id = ServicesIdStrategy.id(entry.category),
        title = entry.category.label,
        shortDescription = shortDescription,
        mainSwitchSettings = mainSwitchSettings,
        content = content,
        dependantSwitchSettings = dependantSwitchSettings,
    )

    internal constructor(
        service: LegacyService,
        mainSwitchSettings: PredefinedUISwitchSettingsUI?,
        content: PredefinedUICardContent?
    ) : this(
        id = ServicesIdStrategy.id(service),
        title = service.name,
        shortDescription = service.categoryLabel,
        mainSwitchSettings = mainSwitchSettings,
        content = content,
    )

    internal constructor(
        tcfHolder: TCFHolder,
        cardContent: PredefinedUICardContent?,
        switchSettings: List<PredefinedUISwitchSettingsUI>? = null,
    ) : this(
        id = tcfHolder.id,
        title = tcfHolder.title,
        shortDescription = null,
        content = cardContent,
        mainSwitchSettings = tcfHolder.mainSwitchSettings,
        switchSettings = switchSettings,
        dependantSwitchSettings = tcfHolder.dependantSwitchSettings
    )

    constructor(specialPurpose: TCFSpecialPurpose, title: String) : this(
        id = ServicesIdStrategy.id(specialPurpose),
        title = specialPurpose.name,
        shortDescription = null,
        mainSwitchSettings = null,
        content = PredefinedUISimpleCardContent(
            title = title,
            description = specialPurpose.purposeDescription,
            values = specialPurpose.illustrations
        )
    )

    constructor(feature: TCFFeature, title: String) : this(
        id = ServicesIdStrategy.id(feature),
        title = feature.name,
        shortDescription = null,
        mainSwitchSettings = null,
        content = PredefinedUISimpleCardContent(
            title = title,
            description = feature.purposeDescription,
            values = feature.illustrations
        ),
    )
}

data class PredefinedUIControllerIDSettings(
    val label: String,
    val value: String
)

data class PredefinedUISwitchSettingsUI(
    val id: String,
    val label: String? = null,
    val disabled: Boolean,
    val currentValue: Boolean,
) {
    internal constructor(service: LegacyService) : this(
        id = PredefinedUIDecision.CONSENT_ID,
        disabled = service.isEssential,
        currentValue = service.consent.status
    )

    internal constructor(tcfHolder: TCFHolder) : this(
        id = PredefinedUIDecision.CONSENT_ID,
        currentValue = tcfHolder.consentValue,
        disabled = false,
    )
}

sealed class PredefinedUICardContent

class PredefinedUISimpleCardContent(
    val title: String = "",
    val description: String = "",
    val value: String,
) : PredefinedUICardContent() {

    constructor(title: String, description: String, values: List<String>) : this(
        title = title,
        description = description,
        value = values.joinToString(separator = PredefinedUIData.CONTENT_SEPARATOR) { illustration ->
            "${PredefinedUIData.BULLET_POINT} $illustration"
        }
    )
}

data class PredefinedUIServicesCardContent(
    val services: List<PredefinedUIServiceDetails>,
) : PredefinedUICardContent()

data class PredefinedUIPurposeCardContent(
    val examples: PredefinedUISimpleCardContent,
    val vendors: PredefinedUIPurposeVendorDetails
) : PredefinedUICardContent()

data class PredefinedUIPurposeVendorDetails(
    val title: String,
    val value: String = ""
)

data class PredefinedUISingleServiceCardContent(
    val service: PredefinedUIServiceDetails,
) : PredefinedUICardContent()

data class PredefinedUIServiceDetails(
    val id: String = "",
    val mainSwitchSettings: PredefinedUISwitchSettingsUI? = null,
    val serviceContentSection: List<PredefinedUIServiceContentSection>? = null,

    val name: String = "",
    val dataCollected: List<String> = listOf(),
    val dataDistribution: PredefinedUIDataDistribution? = null,
    val dataPurposes: List<String> = listOf(),
    val dataRecipients: List<String> = listOf(),
    val serviceDescription: String = "",
    val processingCompany: PredefinedUIProcessingCompany? = null,
    val retentionPeriodDescription: String = "",
    val technologiesUsed: List<String> = listOf(),
    val urls: PredefinedUIURLs? = null,
    val categoryLabel: String = "",
    val consent: PredefinedUIServiceConsent? = null,
    val storageInformationContentSection: PredefinedUIServiceContentSection? = null,
    val dpsDisplayFormat: DpsDisplayFormat? = null,

    private val _legalBasis: List<String> = listOf(),
    private val disableLegalBasis: Boolean = SERVICE_DEFAULT_DISABLE_LEGAL_BASIS,
) {

    internal constructor(
        service: LegacyService,
        storageInformationServiceContent: PredefinedUIServiceContentSection? = null,
        hideTogglesForServices: Boolean = false,
        dpsDisplayFormat: DpsDisplayFormat? = null,
        consent: PredefinedUIServiceConsent,
    ) : this(
        service = service,
        mainSwitchSettings = if (hideTogglesForServices) {
            null
        } else {
            PredefinedUISwitchSettingsUI(
                id = PredefinedUIDecision.CONSENT_ID,
                disabled = service.isEssential,
                currentValue = service.consent.status
            )
        },
        storageInformationServiceContent = storageInformationServiceContent,
        dpsDisplayFormat = dpsDisplayFormat,
        consent = consent,
    )

    internal constructor(
        service: LegacyService,
        mainSwitchSettings: PredefinedUISwitchSettingsUI?,
        storageInformationServiceContent: PredefinedUIServiceContentSection? = null,
        dpsDisplayFormat: DpsDisplayFormat? = null,
        consent: PredefinedUIServiceConsent,
    ) : this(
        id = ServicesIdStrategy.id(service),
        mainSwitchSettings = mainSwitchSettings,
        name = service.name,
        dataCollected = service.dataCollected,
        dataDistribution = service.dataDistribution,
        dataPurposes = service.dataPurposes,
        dataRecipients = service.dataRecipients,
        serviceDescription = service.serviceDescription,
        _legalBasis = service.legalBasis,
        processingCompany = service.processingCompany,
        retentionPeriodDescription = service.retentionPeriodDescription,
        technologiesUsed = service.technologiesUsed,
        urls = service.urls,
        categoryLabel = service.categoryLabel,
        consent = consent,
        storageInformationContentSection = storageInformationServiceContent,
        disableLegalBasis = service.disableLegalBasis,
        dpsDisplayFormat = dpsDisplayFormat
    )

    internal constructor(adTechProvider: AdTechProvider) : this(
        id = ServicesIdStrategy.id(adTechProvider),
        urls = PredefinedUIURLs(privacyPolicy = adTechProvider.privacyPolicyUrl),
    )

    val legalBasis: List<String>
        get() {
            return if (disableLegalBasis) listOf() else _legalBasis
        }
}

data class PredefinedUIDataDistribution(
    val processingLocation: String,
    val thirdPartyCountries: String
)

data class PredefinedUIProcessingCompany(
    val address: String,
    val dataProtectionOfficer: String,
    val name: String
)

data class PredefinedUIServiceContentSection(
    val title: String,
    val content: PredefinedUIServiceContent,
)

data class PredefinedUIURLs(
    val cookiePolicy: String = "",
    val dataProcessingAgreement: String = "",
    val optOut: String = "",
    val privacyPolicy: String = ""
)

sealed class PredefinedUIServiceContent

data class PredefinedUISimpleServiceContent(
    val content: String,
) : PredefinedUIServiceContent()

data class PredefinedUIHyperlinkServiceContent(
    val url: String
) : PredefinedUIServiceContent()

class PredefinedUIStorageInformationServiceContent(
    val content: String,
    val button: PredefinedUIStorageInformationButtonInfo?,
) : PredefinedUIServiceContent()

data class PredefinedUIStorageInformationButtonInfo(
    val label: String,
    val url: String? = null,
    val deviceStorage: List<PredefinedUIDeviceStorageContent>? = null,
)

data class PredefinedUIDeviceStorageContent(
    val title: String,
    val content: List<String>
)

data class PredefinedUIDependantSwitchSettings(
    val id: String,
    val switchSettings: PredefinedUISwitchSettingsUI
) {
    internal constructor(service: LegacyService) : this(
        id = ServicesIdStrategy.id(service),
        switchSettings = PredefinedUISwitchSettingsUI(service)
    )
}

enum class PredefinedUILinkType(val tabIndex: Int?) {
    URL(null),
    MANAGE_SETTINGS(0),
    VENDOR_LIST(1)
}

enum class PredefinedUIButtonType {
    ACCEPT_ALL,
    DENY_ALL,
    OK,
    SAVE_SETTINGS,
    MANAGE_SETTINGS
}

enum class PredefinedUIHtmlLinkType(private val url: String) {
    ACCEPT_ALL_LINK("javascript:UC_UI.acceptAllConsents().then(UC_UI.closeCMP);"),
    DENY_ALL_LINK("javascript:UC_UI.denyAllConsents().then(UC_UI.closeCMP);"),
    SHOW_SECOND_LAYER("javascript:UC_UI.showSecondLayer()");

    companion object {
        fun from(url: String): PredefinedUIHtmlLinkType? = values().find { it.url.equals(url, ignoreCase = true) }
    }
}

enum class PredefinedUIVariant {
    SECOND_LAYER,
    FULL,
    SHEET,
    POPUP_CENTER,
    POPUP_BOTTOM,
}

data class PredefinedUIServiceConsent(
    val history: List<PredefinedUIHistoryEntry>,
    val status: Boolean,
)

data class PredefinedUIHistoryEntry(
    val status: Boolean,
    val decisionText: String,
    val formattedDate: String,
)
