package com.usercentrics.sdk.models.settings

import com.usercentrics.sdk.AdTechProvider
import com.usercentrics.sdk.UserDecision
import com.usercentrics.sdk.models.settings.serviceType.BaseServiceType
import com.usercentrics.sdk.models.settings.serviceType.GDPRServiceType
import com.usercentrics.sdk.models.settings.serviceType.TCFServiceType
import com.usercentrics.sdk.services.tcf.interfaces.*
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory

internal class ServicesIdStrategy {

    companion object {

        fun id(category: UsercentricsCategory): String {
            return "${GDPRServiceType.CATEGORY.prefix}=${category.categorySlug}"
        }

        fun id(service: LegacyService): String {
            return "${GDPRServiceType.SERVICE.prefix}=${service.id}"
        }

        fun id(vendor: TCFVendor): String {
            return "${TCFServiceType.VENDOR.prefix}=${vendor.id}"
        }

        fun id(stack: TCFStack): String {
            return "${TCFServiceType.STACK.prefix}=${stack.id}"
        }

        fun id(specialFeature: TCFSpecialFeature): String {
            return "${TCFServiceType.SPECIAL_FEATURE.prefix}=${specialFeature.id}"
        }

        fun id(purpose: TCFPurpose): String {
            return "${TCFServiceType.PURPOSE.prefix}=${purpose.id}"
        }

        fun id(specialPurpose: TCFSpecialPurpose): String {
            return "${TCFServiceType.SPECIAL_PURPOSE.prefix}=${specialPurpose.id}"
        }

        fun id(feature: TCFFeature): String {
            return "${TCFServiceType.FEATURE.prefix}=${feature.id}"
        }

        fun id(adTechProvider: AdTechProvider): String {
            return "${TCFServiceType.AD_TECH_PROVIDER.prefix}=${adTechProvider.id}"
        }

        fun userDecisionsGDPR(userDecisions: List<PredefinedUIDecision>): List<UserDecision> {
            return userDecisions
                .filter { userDecision -> isGDPRDecision(userDecision.serviceId) }
                .mapNotNull { userDecision ->
                    val consentStatus = userDecision.consent()
                    if (consentStatus != null) UserDecision(
                        serviceId = actualServiceId(userDecision.serviceId),
                        consent = consentStatus
                    ) else null
                }
        }

        fun userDecisionsTCF(userDecisions: List<PredefinedUIDecision>): TCFUserDecisions {
            val tcfDecisions = userDecisions.filter { userDecision -> isTCFDecision(userDecision.serviceId) }
            if (tcfDecisions.isEmpty()) {
                return TCFUserDecisions(
                    purposes = emptyList(),
                    specialFeatures = emptyList(),
                    vendors = emptyList(),
                    adTechProviders = emptyList(),
                )
            }

            val purposes = mutableListOf<TCFUserDecisionOnPurpose>()
            val specialFeatures = mutableListOf<TCFUserDecisionOnSpecialFeature>()
            val vendors = mutableListOf<TCFUserDecisionOnVendor>()
            val adTechProviders = mutableListOf<AdTechProviderDecision>()

            tcfDecisions.forEach { userDecision ->
                val actualServiceId = actualServiceId(userDecision.serviceId).toInt()

                when (tcfServiceType(userDecision.serviceId)) {
                    TCFServiceType.VENDOR -> {
                        vendors.add(TCFUserDecisionOnVendor(actualServiceId, userDecision.consent(), userDecision.legitimateInterest()))
                    }

                    TCFServiceType.SPECIAL_FEATURE -> {
                        specialFeatures.add(TCFUserDecisionOnSpecialFeature(actualServiceId, userDecision.consent()))
                    }

                    TCFServiceType.PURPOSE -> {
                        purposes.add(TCFUserDecisionOnPurpose(actualServiceId, userDecision.consent(), userDecision.legitimateInterest()))
                    }

                    TCFServiceType.AD_TECH_PROVIDER -> {
                        adTechProviders.add(AdTechProviderDecision(actualServiceId, userDecision.consent() ?: false))
                    }

                    TCFServiceType.SPECIAL_PURPOSE,
                    TCFServiceType.FEATURE,
                    TCFServiceType.STACK,
                    null -> {
                    }
                }
            }

            return TCFUserDecisions(
                purposes = purposes,
                specialFeatures = specialFeatures,
                vendors = vendors,
                adTechProviders = adTechProviders,
            )
        }

        private fun isTCFDecision(serviceId: String): Boolean {
            return tcfServiceType(serviceId) != null
        }

        private fun tcfServiceType(serviceId: String): TCFServiceType? {
            return TCFServiceType.values().firstOrNull { serviceType -> matchesServiceType(serviceId, serviceType) }
        }

        private fun isGDPRDecision(serviceId: String): Boolean {
            return GDPRServiceType.values().any { serviceType -> matchesServiceType(serviceId, serviceType) }
        }

        private fun matchesServiceType(serviceId: String, serviceType: BaseServiceType): Boolean {
            return serviceId.startsWith(serviceType.prefix)
        }

        private fun actualServiceId(serviceId: String): String {
            return serviceId.split('=').getOrElse(1) { error("invalid id") }
        }
    }
}
