package com.usercentrics.sdk.services.billing

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.services.api.BillingApi
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher

internal interface BillingService {

    fun reportSession(settingsId: String)
    fun dispatchSessionBuffer()
}

internal class BillingServiceImpl(
    private val dispatcher: Dispatcher,
    private val storageInstance: DeviceStorage,
    private val billingApi: BillingApi
) : BillingService {

    companion object {
        private const val BILLING_PERIOD_IN_DAYS = 1
    }

    override fun reportSession(settingsId: String) {
        if (shouldReportNewSession()) {
            reportSession(settingsId, timestamp = DateTime().timestamp())
        }
    }

    override fun dispatchSessionBuffer() {
        dispatcher.dispatch {
            storageInstance.getAndEraseSessionBuffer().forEach {
                reportSession(it.settingsId, it.timestamp)
            }
        }
    }

    private fun shouldReportNewSession(): Boolean {
        val lastSessionTimestamp = storageInstance.getSessionTimestamp() ?: return true
        val lastSessionDateTimeAtMidnight = DateTime(lastSessionTimestamp).atMidnight()

        val currentDateTimeAtMidnight = DateTime().atMidnight()
        val billingReportDeltaInDays = currentDateTimeAtMidnight.diffInDays(lastSessionDateTimeAtMidnight)
        return billingReportDeltaInDays >= BILLING_PERIOD_IN_DAYS
    }

    private fun reportSession(settingsId: String, timestamp: Long) {
        try {
            billingApi.report(settingsId)
            storageInstance.setSessionTimestamp(timestamp)
        } catch (error: Throwable) {
            dispatcher.dispatch {
                storageInstance.addSessionToBuffer(timestamp, settingsId)
            }
        }
    }
}
