package com.usercentrics.sdk.services.deviceStorage

import com.usercentrics.ccpa.CCPAStorage
import com.usercentrics.ccpa.CcpaApi
import com.usercentrics.sdk.assertNotUIThread
import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.api.ApiConstants
import com.usercentrics.sdk.models.common.UserSessionDataConsent
import com.usercentrics.sdk.models.settings.LegacyExtendedSettings
import com.usercentrics.sdk.models.settings.LegacyService
import com.usercentrics.sdk.models.settings.SettingsVersion
import com.usercentrics.sdk.services.deviceStorage.migrations.Migration
import com.usercentrics.sdk.services.deviceStorage.migrations.MigrationException
import com.usercentrics.sdk.services.deviceStorage.migrations.MigrationNotFoundException
import com.usercentrics.sdk.services.deviceStorage.migrations.MigrationToVersion1
import com.usercentrics.sdk.services.deviceStorage.models.*
import com.usercentrics.tcf.core.IABTCFKeys
import com.usercentrics.tcf.core.model.gvl.MAX_PURPOSE_ID

internal class UsercentricsDeviceStorage private constructor(
    private val storageHolder: StorageHolder,
    private val logger: UsercentricsLogger,
    private val currentVersion: Int,
    private val migrations: List<Migration>,
    private val jsonParser: JsonParser
) : DeviceStorage {

    private val defaultStorage = storageHolder.defaultKeyValueStorage
    private val usercentricsStorage = storageHolder.usercentricsKeyValueStorage

    private var settings: StorageSettings = StorageSettings()
    private var tcfData: StorageTCF = StorageTCF()

    override fun init() {
        runMigrations()
    }

    override fun clear() {
        logger.debug("Clearing local storage")

        clearUsercentricsStorageEntries()
        clearTCFStorageEntries()
        clearCCPAStorageEntries()

        this.settings = StorageSettings()
        this.tcfData = StorageTCF()
    }

    override fun getActualTCFSettingsId(): String {
        return usercentricsStorage.getString(StorageKeys.ACTUAL_TCF_SETTINGS_ID.text, "") ?: ""
    }

    override fun storeValuesDefaultStorage(values: Map<String, Any>) {
        defaultStorage.putValuesMap(values)
    }

    override fun toCcpaStorage(): CCPAStorage {
        return this.defaultStorage.toCcpaStorage()
    }

    override fun setCcpaTimestampInMillis(timestampInMillis: Long) {
        usercentricsStorage.put(
            StorageKeys.CCPA_TIMESTAMP.text,
            timestampInMillis.toString()
        )
    }

    override fun getCcpaTimestampInMillis(): Long? {
        return try {
            usercentricsStorage.getString(StorageKeys.CCPA_TIMESTAMP.text, null)?.toLong()
        } catch (e: Throwable) {
            null
        }
    }

    override fun setSessionTimestamp(sessionTimestamp: Long) {
        usercentricsStorage.put(
            StorageKeys.SESSION_TIMESTAMP.text,
            sessionTimestamp.toString()
        )
    }

    override fun getSessionTimestamp(): Long? {
        val timestamp = usercentricsStorage.getString(StorageKeys.SESSION_TIMESTAMP.text, null)
        return try {
            timestamp?.toLong()
        } catch (e: Throwable) {
            null
        }
    }

    override fun bootSettings(settingsId: String) {
        var parsedSettings: StorageSettings? = null

        val storageSettings = usercentricsStorage.getString(StorageKeys.SETTINGS_PATTERN.text + settingsId, null)
        if (!storageSettings.isNullOrBlank()) {
            parsedSettings = jsonParser.tryToDecodeFromString(StorageSettings.serializer(), storageSettings, logger)
        }
        this.settings = parsedSettings ?: StorageSettings()
    }

    override fun fetchSettings(): StorageSettings {
        return this.settings
    }

    override fun lastInteractionTimestamp(): Long? {
       return fetchSettings().lastInteractionTimestamp
    }

    override fun getSettingsVersion(): String {
        return fetchSettings().version
    }

    override fun getSettingsId(): String {
        return fetchSettings().id
    }

    override fun getControllerId(): String {
        return fetchSettings().controllerId
    }

    override fun getSettingsLanguage(): String {
        return fetchSettings().language
    }

    override fun getUserSessionDataConsents(): List<UserSessionDataConsent> {
        val consentList = mutableListOf<UserSessionDataConsent>()
        val settings = this.fetchSettings()

        settings.services.forEach { service ->
            service.history.forEach { history ->
                consentList.add(
                    UserSessionDataConsent(
                        status = history.status,
                        templateId = service.id,
                        timestampInMillis = history.timestampInMillis,
                    )
                )
            }
        }

        return consentList
    }

    override fun saveSettings(settings: LegacyExtendedSettings, services: List<LegacyService>) {
        val versionChangeRequiresReshow = shouldReshowBannerAfterVersionChange(settings)
        if (versionChangeRequiresReshow) {
            this.usercentricsStorage.put(StorageKeys.USER_ACTION_REQUIRED.text, true.toString())
        }

        val storageSettings = mapStorageSettings(settings, services)
        this.settings = storageSettings
        usercentricsStorage.put(
            key = StorageKeys.SETTINGS_PATTERN.text + settings.id,
            value = jsonParser.encodeToString(StorageSettings.serializer(), storageSettings)
        )
    }

    private fun clearUsercentricsStorageEntries() {
        usercentricsStorage.deleteAll(exceptions = listOf(StorageKeys.LOCATION_CACHE.text))
    }

    override fun clearTCFStorageEntries() {
        enumValues<IABTCFKeys>().forEach { entry ->
            defaultStorage.deleteKey(entry.key)
        }

        for (i in 1..MAX_PURPOSE_ID) {
            defaultStorage.deleteKey(IABTCFKeys.publisherRestrictionsKeyOf(i))
        }
    }

    private fun mapStorageSettings(settings: LegacyExtendedSettings, services: List<LegacyService>): StorageSettings {
        val selectedLanguage = if (settings.isTcfEnabled) {
            settings.tcfui?.language?.selected!!
        } else {
            settings.ui?.language?.selected!!
        }

        return StorageSettings(
            controllerId = settings.controllerId,
            id = settings.id,
            language = selectedLanguage.isoCode,
            services = mapStorageServices(services),
            version = settings.version,
        )
    }

    private fun mapStorageServices(services: List<LegacyService>): List<StorageService> {
        return services.map { service ->
            return@map StorageService(
                history = service.consent.history.map { StorageConsentHistory.fromConsentHistory(it) },
                id = service.id,
                processorId = service.processorId,
                status = service.consent.status
            )
        }
    }

    override fun getConsentBuffer(): ConsentsBuffer {
        assertNotUIThread()
        val bufferString = usercentricsStorage.getString(StorageKeys.CONSENTS_BUFFER.text, null) ?: ""
        return jsonParser.tryToDecodeFromString(ConsentsBuffer.serializer(), bufferString) ?: ConsentsBuffer(emptyList())
    }

    override fun setConsentBuffer(buffer: ConsentsBuffer) {
        assertNotUIThread()
        usercentricsStorage.put(
            StorageKeys.CONSENTS_BUFFER.text,
            jsonParser.encodeToString(ConsentsBuffer.serializer(), buffer)
        )
    }

    override fun bootTCFData(settingsId: String): StorageTCF {
        val tcfData = this.usercentricsStorage.getString(StorageKeys.TCF_PATTERN.text + settingsId, null) ?: ""
        if (tcfData.isNotBlank()) {
            jsonParser.tryToDecodeFromString(StorageTCF.serializer(), tcfData, logger)?.let {
                this.tcfData = it
            }
        }
        return this.tcfData
    }

    override fun saveTCFData(tcfData: StorageTCF) {
        this.tcfData = tcfData

        val settingsId = this.settings.id

        this.usercentricsStorage.put(StorageKeys.TCF_PATTERN.text + settingsId, jsonParser.encodeToString(StorageTCF.serializer(), tcfData))
        saveActualTCFSettingsId(settingsId)
    }

    override fun saveActualTCFSettingsId(actualSettingsId: String) {
        this.usercentricsStorage.put(StorageKeys.ACTUAL_TCF_SETTINGS_ID.text, actualSettingsId)
    }

    override fun getTCFData(): StorageTCF {
        return this.tcfData
    }

    override fun fetchCcpaString(): String {
        return defaultStorage.getString(CcpaApi.privacyStringStorageKey, null) ?: ""
    }

    override fun addSessionToBuffer(currentTime: Long, settingsId: String) {
        val sessionBufferSet = readSessionBuffer().toMutableSet()
        sessionBufferSet.add(StorageSessionEntry(settingsId, currentTime))

        writeSessionBuffer(sessionBufferSet)
    }

    override fun getAndEraseSessionBuffer(): List<StorageSessionEntry> {
        val sessionBuffer = readSessionBuffer()
        clearSessionBuffer()
        return sessionBuffer
    }

    override fun saveABTestingVariant(variant: String) {
        this.usercentricsStorage.put(StorageKeys.AB_TESTING_VARIANT.text, variant)
    }

    override fun getABTestingVariant(): String? {
        return this.usercentricsStorage.getString(StorageKeys.AB_TESTING_VARIANT.text, null)
    }

    override fun getUserActionRequired(): Boolean {
        val userActionRequired = this.usercentricsStorage.getString(StorageKeys.USER_ACTION_REQUIRED.text, null)
        return userActionRequired?.toBoolean() == true
    }

    override fun clearUserActionRequired() {
        this.usercentricsStorage.deleteKey(StorageKeys.USER_ACTION_REQUIRED.text)
    }

    override fun saveACString(acString: String) {
        defaultStorage.put(IABTCFKeys.ADDITIONAL_CONSENT_MODE.key, acString)
    }

    override fun getACString(): String {
        return defaultStorage.getString(IABTCFKeys.ADDITIONAL_CONSENT_MODE.key, null) ?: ""
    }

    override fun deleteSettingsThatDoNotMatch(settingsIds: Set<String>) {
        this.usercentricsStorage.deleteKeysThatDoNotMatch(pattern = StorageKeys.TCF_PATTERN.text, values = settingsIds)
        this.usercentricsStorage.deleteKeysThatDoNotMatch(pattern = StorageKeys.SETTINGS_PATTERN.text, values = settingsIds)
    }

    private fun shouldReshowBannerAfterVersionChange(currentSettings: LegacyExtendedSettings): Boolean {
        val storageVersion = this.settings.version
        if (storageVersion.isBlank()) {
            return false
        }

        if (currentSettings.showFirstLayerOnVersionChange.isEmpty()) {
            return false
        }

        val currentVersionArray = currentSettings.version.split('.')
        val storageVersionArray = storageVersion.split('.')

        return ((SettingsVersion.MAJOR.ordinal in currentSettings.showFirstLayerOnVersionChange && currentVersionArray[0] != storageVersionArray[0]) ||
                (SettingsVersion.MINOR.ordinal in currentSettings.showFirstLayerOnVersionChange && currentVersionArray[1] != storageVersionArray[1]) ||
                (SettingsVersion.PATCH.ordinal in currentSettings.showFirstLayerOnVersionChange && currentVersionArray[2] != storageVersionArray[2]))
    }

    private fun readSessionBuffer(): List<StorageSessionEntry> {
        assertNotUIThread()
        val jsonArray = usercentricsStorage.getString(StorageKeys.SESSION_BUFFER.text, null)
        if (jsonArray.isNullOrBlank()) {
            return listOf()
        }
        return jsonParser.decodeFromString(jsonArray)
    }

    private fun clearSessionBuffer() {
        writeSessionBuffer(setOf())
    }

    private fun writeSessionBuffer(sessionBufferSet: Set<StorageSessionEntry>) {
        usercentricsStorage.put(StorageKeys.SESSION_BUFFER.text, jsonParser.encodeToString(sessionBufferSet))
    }

    private fun runMigrations() {
        val storageVersion = getStorageVersion()
        if (shouldMigrate(storageVersion)) {
            ((storageVersion + 1)..currentVersion).forEach { targetVersion ->
                val oldVersion = targetVersion - 1
                try {
                    migrateDataAfterVersionChange(oldVersion, targetVersion)
                } catch (cause: Throwable) {
                    throw MigrationException("Cannot migrate stored data from $oldVersion to $targetVersion", cause)
                }
            }
        }
        saveStorageCurrentVersion()
    }

    private fun shouldMigrate(storageVersion: Int): Boolean {
        return if (storageVersion == 0) {
            hasDataFromVersion0()
        } else {
            storageVersion < currentVersion
        }
    }

    private fun hasDataFromVersion0(): Boolean {
        return enumValues<MigrationToVersion1.V0StorageKeys>().any { key ->
            storageHolder.defaultKeyValueStorage.hasKey(key.text)
        }
    }

    private fun getStorageVersion(): Int {
        return usercentricsStorage.getNumber(
            StorageKeys.STORAGE_VERSION.text,
            ApiConstants.STORAGE_DEFAULT_VERSION
        )
    }

    @Suppress("FoldInitializerAndIfToElvis")
    private fun migrateDataAfterVersionChange(oldVersion: Int, targetVersion: Int) {
        val hasExactMigration = migrations.firstOrNull { it.fromVersion == oldVersion && it.toVersion == targetVersion }
        if (hasExactMigration == null) {
            throw MigrationNotFoundException(oldVersion, targetVersion)
        }

        migrations.forEach { migration ->
            if (migration.fromVersion == oldVersion && migration.toVersion == targetVersion) {
                migration.migrate()
            }
        }
    }

    private fun saveStorageCurrentVersion() {
        usercentricsStorage.put(StorageKeys.STORAGE_VERSION.text, currentVersion)
    }

    private fun clearCCPAStorageEntries() {
        defaultStorage.deleteKey(CcpaApi.privacyStringStorageKey)
    }

    internal class Builder(
        private val storageHolder: StorageHolder,
        private val logger: UsercentricsLogger,
        private val jsonParser: JsonParser,
        private val currentVersion: Int = ApiConstants.CURRENT_STORAGE_VERSION
    ) {

        private val migrations: MutableList<Migration> = mutableListOf()

        fun addMigration(vararg migration: Migration): Builder {
            migrations.addAll(migration)
            return this
        }

        fun build(): DeviceStorage {
            val deviceStorage = UsercentricsDeviceStorage(
                storageHolder = storageHolder,
                logger = logger,
                currentVersion = currentVersion,
                migrations = migrations,
                jsonParser = jsonParser
            )
            deviceStorage.init()
            return deviceStorage
        }
    }
}
