package com.usercentrics.sdk.services.deviceStorage.migrations

import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.extensions.secondsToMillis
import com.usercentrics.sdk.models.settings.UsercentricsConsentAction
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import com.usercentrics.sdk.services.deviceStorage.StorageHolder
import com.usercentrics.sdk.services.deviceStorage.models.StorageConsentAction
import com.usercentrics.sdk.services.deviceStorage.models.StorageConsentHistory
import com.usercentrics.sdk.services.deviceStorage.models.StorageConsentType
import com.usercentrics.sdk.services.deviceStorage.models.StorageService
import com.usercentrics.sdk.services.deviceStorage.models.StorageSettings
import kotlinx.serialization.json.*

internal abstract class AbstractMigrationSettingsV2(
    version: Int,
    storageHolder: StorageHolder,
    private val json: JsonParser,
):Migration(storageHolder, version) {

    protected fun storageSettingsFromCache(settingsValue: String): StorageSettings {
        val settingsJsonObject = json.decodeFromString(JsonObject.serializer(), settingsValue)

        val servicesArray = settingsJsonObject["services"]!!.jsonArray
        val services: List<StorageService> = servicesArray.map {
            val serviceJsonObject = it.jsonObject
            StorageService(
                history = settingsHistoryFromServiceJson(serviceJsonObject),
                id = serviceJsonObject["id"]!!.jsonPrimitive.content,
                processorId = serviceJsonObject["processorId"]!!.jsonPrimitive.content,
                status = serviceJsonObject["status"]!!.jsonPrimitive.boolean
            )
        }

        return StorageSettings(
            controllerId = settingsJsonObject["controllerId"]!!.jsonPrimitive.content,
            id = settingsJsonObject["id"]!!.jsonPrimitive.content,
            language = settingsJsonObject["language"]!!.jsonPrimitive.content,
            services = services,
            version = settingsJsonObject["version"]!!.jsonPrimitive.content
        )
    }

    private fun settingsHistoryFromServiceJson(serviceJsonObject: JsonObject): List<StorageConsentHistory> {
        return serviceJsonObject["history"]!!.jsonArray.map { historyJson ->
            val historyJsonObject = historyJson.jsonObject

            val timestampFieldRaw = historyJsonObject["timestamp"]?.jsonPrimitive
            val timestampInMillisFieldRaw = historyJsonObject["timestampInMillis"]?.jsonPrimitive

            val timestampInSeconds = timestampFieldRaw?.double ?: timestampInMillisFieldRaw?.double!!
            val timestampInMillis = timestampInSeconds.toLong().secondsToMillis()

            val consentAction = UsercentricsConsentAction.valueOf(historyJsonObject["action"]!!.jsonPrimitive.content)
            val consentType = UsercentricsConsentType.valueOf(historyJsonObject["type"]!!.jsonPrimitive.content)

            StorageConsentHistory(
                action = StorageConsentAction.fromConsentAction(consentAction),
                status = historyJsonObject["status"]!!.jsonPrimitive.boolean,
                type = StorageConsentType.fromConsentType(consentType),
                language = historyJsonObject["language"]!!.jsonPrimitive.content,
                timestampInMillis = timestampInMillis
            )
        }
    }
}
