package com.usercentrics.sdk.services.deviceStorage.migrations

import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.extensions.secondsToMillis
import com.usercentrics.sdk.models.settings.UsercentricsConsentAction
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import com.usercentrics.sdk.services.deviceStorage.StorageHolder
import com.usercentrics.sdk.services.deviceStorage.StorageKeys
import com.usercentrics.sdk.services.deviceStorage.models.*
import kotlinx.serialization.json.*

internal class MigrationToVersion1(
    storageHolder: StorageHolder,
    private val jsonParser: JsonParser
) : Migration(storageHolder, 1) {

    override fun migrate() {
        migrateFromDefaultStorageToCustom(oldKey = V0StorageKeys.CCPA_TIMESTAMP.text, newKey = "ccpa_timestamp_millis")
        migrateFromDefaultStorageToCustom(oldKey = V0StorageKeys.SESSION_TIMESTAMP.text, newKey = "session_timestamp")
        migrateFromDefaultStorageToCustom(oldKey = V0StorageKeys.CONSENTS_BUFFER.text, newKey = "consents_buffer")
        migrateFromDefaultStorageToCustom(oldKey = V0StorageKeys.TCF.text, newKey = "tcf")
        migrateSettings()

        clear()
    }

    private fun migrateFromDefaultStorageToCustom(oldKey: String, newKey: String) {
        val value = getLegacyValue(oldKey)
        if (value?.isNotBlank() == true) {
            storeNewValue(newKey, value)
        }
    }

    private fun getLegacyValue(oldKey: String): String? {
        return storageHolder.defaultKeyValueStorage.getString(oldKey, null)
    }

    private fun storeNewValue(newKey: String, value: String) {
        storageHolder.usercentricsKeyValueStorage.put(newKey, value)
    }

    private fun migrateSettings() {
        val legacyValue = getLegacyValue(V0StorageKeys.SETTINGS.text)
        if (legacyValue.isNullOrBlank()) {
            return
        }
        val newSettings = storageSettingsFromLegacyJson(legacyValue)
        storeNewValue("settings", jsonParser.encodeToString(newSettings))
    }

    private fun storageSettingsFromLegacyJson(legacyValue: String): StorageSettings {
        val settingsJsonObject = jsonParser.decodeFromString(JsonObject.serializer(), legacyValue)

        val servicesArray = settingsJsonObject["services"]!!.jsonArray
        val services: List<StorageService> = servicesArray.map {
            val serviceJsonObject = it.jsonObject
            StorageService(
                history = settingsHistoryFromServiceJson(serviceJsonObject),
                id = serviceJsonObject["id"]!!.jsonPrimitive.content,
                processorId = serviceJsonObject["processorId"]!!.jsonPrimitive.content,
                status = serviceJsonObject["status"]!!.jsonPrimitive.boolean
            )
        }

        return StorageSettings(
            controllerId = settingsJsonObject["controllerId"]!!.jsonPrimitive.content,
            id = settingsJsonObject["id"]!!.jsonPrimitive.content,
            language = settingsJsonObject["language"]!!.jsonPrimitive.content,
            services = services,
            version = settingsJsonObject["version"]!!.jsonPrimitive.content
        )
    }

    private fun settingsHistoryFromServiceJson(serviceJsonObject: JsonObject): List<StorageConsentHistory> {
        return serviceJsonObject["history"]!!.jsonArray.map { historyJson ->
            val historyJsonObject = historyJson.jsonObject

            val timestampInSeconds = historyJsonObject["timestamp"]!!.jsonPrimitive.double
            val timestampInMillis = timestampInSeconds.toLong().secondsToMillis()

            val consentAction = UsercentricsConsentAction.valueOf(historyJsonObject["action"]!!.jsonPrimitive.content)
            val consentType = UsercentricsConsentType.valueOf(historyJsonObject["type"]!!.jsonPrimitive.content)

            StorageConsentHistory(
                action = StorageConsentAction.fromConsentAction(consentAction),
                status = historyJsonObject["status"]!!.jsonPrimitive.boolean,
                type = StorageConsentType.fromConsentType(consentType),
                language = historyJsonObject["language"]!!.jsonPrimitive.content,
                timestampInMillis = timestampInMillis
            )
        }
    }

    private fun clear() {
        enumValues<V0StorageKeys>().forEach { key ->
            storageHolder.defaultKeyValueStorage.deleteKey(key.text)
        }
    }

    internal enum class V0StorageKeys(val text: String) {
        CACHE_KEY("uc_cache"),
        CCPA_TIMESTAMP("uc_ccpa"),
        CMP_ID("CMP-ID"),
        CONSENTS_BUFFER("uc_consents_buffer"),
        SESSION_TIMESTAMP("uc_session_timestamp"),
        SETTINGS("uc_settings"),
        TCF("uc_tcf"),
        SESSION_BUFFER("uc_session_buffer"),
        USER_INTERACTION("uc_user_interaction")
    }
}
