package com.usercentrics.sdk.services.deviceStorage.models

import com.usercentrics.sdk.models.settings.UsercentricsConsentAction
import com.usercentrics.sdk.models.settings.LegacyConsentHistoryEntry
import com.usercentrics.sdk.models.settings.UsercentricsConsentType
import kotlinx.serialization.Serializable

@Serializable
internal data class StorageSettings(
    val controllerId: String = "",
    val id: String = "",
    val language: String = "",
    val services: List<StorageService> = listOf(),
    val version: String = "",
) {

    val lastInteractionTimestamp: Long?
        get() = services.mapNotNull { it.lastInteractionTimestamp }.maxOfOrNull { it }

}

@Serializable
internal data class StorageService(
    val history: List<StorageConsentHistory>,
    val id: String,
    val processorId: String,
    val status: Boolean
) {
    val lastInteractionTimestamp: Long?
        get() {
            return history.filter {
                if (it.action == StorageConsentAction.NON_EU_REGION) {
                    return@filter true
                }
                it.type != StorageConsentType.IMPLICIT
            }.maxOfOrNull {
                it.timestampInMillis
            }
        }
}

@Serializable
internal data class StorageConsentHistory(
    val action: StorageConsentAction,
    val status: Boolean,
    val type: StorageConsentType,
    val language: String,
    val timestampInMillis: Long,
) {
    companion object {
        fun fromConsentHistory(consentHistory: LegacyConsentHistoryEntry): StorageConsentHistory {
            return StorageConsentHistory(
                action = StorageConsentAction.fromConsentAction(consentHistory.action),
                status = consentHistory.status,
                type = StorageConsentType.fromConsentType(consentHistory.type),
                language = consentHistory.language,
                timestampInMillis = consentHistory.timestampInMillis
            )
        }
    }

    fun toConsentHistory(): LegacyConsentHistoryEntry {
        return LegacyConsentHistoryEntry(
            action = action.toConsentAction(),
            status = status,
            type = type.toConsentType(),
            language = language,
            timestampInMillis = timestampInMillis
        )
    }
}

@Serializable
internal enum class StorageConsentAction {
    ACCEPT_ALL_SERVICES,
    DENY_ALL_SERVICES,
    ESSENTIAL_CHANGE,
    INITIAL_PAGE_LOAD,
    NON_EU_REGION,
    SESSION_RESTORED,
    TCF_STRING_CHANGE,
    UPDATE_SERVICES;

    companion object {
        fun fromConsentAction(action: UsercentricsConsentAction): StorageConsentAction {
            return when (action) {
                UsercentricsConsentAction.ACCEPT_ALL_SERVICES -> ACCEPT_ALL_SERVICES
                UsercentricsConsentAction.DENY_ALL_SERVICES -> DENY_ALL_SERVICES
                UsercentricsConsentAction.ESSENTIAL_CHANGE -> ESSENTIAL_CHANGE
                UsercentricsConsentAction.INITIAL_PAGE_LOAD -> INITIAL_PAGE_LOAD
                UsercentricsConsentAction.NON_EU_REGION -> NON_EU_REGION
                UsercentricsConsentAction.SESSION_RESTORED -> SESSION_RESTORED
                UsercentricsConsentAction.TCF_STRING_CHANGE -> TCF_STRING_CHANGE
                UsercentricsConsentAction.UPDATE_SERVICES -> UPDATE_SERVICES
            }
        }
    }

    fun toConsentAction(): UsercentricsConsentAction {
        return when (this) {
            ACCEPT_ALL_SERVICES -> UsercentricsConsentAction.ACCEPT_ALL_SERVICES
            DENY_ALL_SERVICES -> UsercentricsConsentAction.DENY_ALL_SERVICES
            ESSENTIAL_CHANGE -> UsercentricsConsentAction.ESSENTIAL_CHANGE
            INITIAL_PAGE_LOAD -> UsercentricsConsentAction.INITIAL_PAGE_LOAD
            NON_EU_REGION -> UsercentricsConsentAction.NON_EU_REGION
            SESSION_RESTORED -> UsercentricsConsentAction.SESSION_RESTORED
            TCF_STRING_CHANGE -> UsercentricsConsentAction.TCF_STRING_CHANGE
            UPDATE_SERVICES -> UsercentricsConsentAction.UPDATE_SERVICES
        }
    }
}

@Serializable
internal enum class StorageConsentType {
    EXPLICIT,
    IMPLICIT;

    companion object {
        fun fromConsentType(type: UsercentricsConsentType): StorageConsentType {
            return when (type) {
                UsercentricsConsentType.EXPLICIT -> EXPLICIT
                UsercentricsConsentType.IMPLICIT -> IMPLICIT
            }
        }
    }

    fun toConsentType(): UsercentricsConsentType {
        return when (this) {
            EXPLICIT -> UsercentricsConsentType.EXPLICIT
            IMPLICIT -> UsercentricsConsentType.IMPLICIT
        }
    }
}
