package com.usercentrics.sdk.services.initialValues

import com.usercentrics.sdk.acm.service.AdditionalConsentModeService
import com.usercentrics.sdk.core.settings.SettingsOrchestrator
import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.common.InitialView
import com.usercentrics.sdk.models.common.UsercentricsVariant
import com.usercentrics.sdk.models.dataFacade.MergedServicesSettings
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.services.dataFacade.DataFacade
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.services.initialValues.variants.*
import com.usercentrics.sdk.services.settings.ISettingsLegacy
import com.usercentrics.sdk.services.tcf.TCFUseCase
import com.usercentrics.sdk.v2.location.data.UsercentricsLocation
import com.usercentrics.sdk.v2.location.service.ILocationService
import com.usercentrics.sdk.v2.settings.data.CCPARegion

internal class InitialValuesStrategyImpl(
    private val dataFacade: DataFacade,
    private val deviceStorage: DeviceStorage,
    private val settingsLegacy: ISettingsLegacy,
    private val locationService: ILocationService,
    private val tcf: TCFUseCase,
    private val ccpaStrategy: CCPAStrategy,
    private val tcfStrategy: TCFStrategy,
    private val gdprStrategy: GDPRStrategy,
    private val settingsOrchestrator: SettingsOrchestrator,
    private val additionalConsentModeService: AdditionalConsentModeService,
    private val logger: UsercentricsLogger,
) : InitialValuesStrategy {

    companion object {
        private val defaultCCPARegion = CCPARegion.US_CA_ONLY
    }

    private val noShowFlag
        get() = settingsOrchestrator.noShow

    override var variant: UsercentricsVariant? = null

    override suspend fun boot(isFirstInitialization: Boolean, controllerId: String) {
        loadVariant()
        loadConsents(isFirstInitialization, controllerId)
    }

    fun loadVariant() {
        val settings = settingsLegacy.getSettings()
        val location = locationService.location
        variant = resolveVariant(settings, location)
    }

    private fun resolveVariant(settings: LegacyExtendedSettings, location: UsercentricsLocation): UsercentricsVariant {
        val isCCPAEnabled = settings.ccpa?.isActive == true || settings.framework != null
        val isTCFEnabled = settings.isTcfEnabled

        return when {
            isCCPAEnabled -> getVariantForCCPA(settings, location)
            isTCFEnabled -> UsercentricsVariant.TCF
            else -> UsercentricsVariant.DEFAULT
        }
    }

    private fun getVariantForCCPA(settings: LegacyExtendedSettings, location: UsercentricsLocation): UsercentricsVariant {
        return when (settings.ccpa?.region ?: defaultCCPARegion) {
            CCPARegion.US_CA_ONLY -> {
                if (location.isInCalifornia()) {
                    UsercentricsVariant.CCPA
                } else {
                    UsercentricsVariant.DEFAULT
                }
            }

            CCPARegion.US -> {
                if (location.isInUS()) {
                    UsercentricsVariant.CCPA
                } else {
                    UsercentricsVariant.DEFAULT
                }
            }

            CCPARegion.ALL -> UsercentricsVariant.CCPA
        }
    }

    override fun loadConsents(isFirstInitialization: Boolean, controllerId: String) {
        val settings = settingsLegacy.getSettings()
        val location = locationService.location

        if (isFirstInitialization) {
            initializeImplicitConsentTheFirstTime(controllerId, settings, location)
            initializeCCPAStringTheFirstTime(settings)
            return
        }

        val shouldAcceptAllImplicitlyOnInit = shouldAcceptAllImplicitlyOnInit(variant!!, settings, location.isInEU())

        val mergedServicesSettings = mergeSettingsFromStorage(controllerId, shouldAcceptAllImplicitlyOnInit)
        val newServices = mergedServicesSettings?.updatedNonEssentialServices

        if (!newServices.isNullOrEmpty() && shouldAcceptAllImplicitlyOnInit) {
            acceptAllImplicitly(controllerId, newServices)
        }
    }

    private fun initializeImplicitConsentTheFirstTime(controllerId: String, settings: LegacyExtendedSettings, location: UsercentricsLocation) {
        val shouldAcceptAllImplicitlyOnInit = shouldAcceptAllImplicitlyOnInit(variant!!, settings, location.isInEU())
        if (shouldAcceptAllImplicitlyOnInit) {
            acceptAllImplicitly(controllerId)
        } else {
            denyAllImplicitly(controllerId)
        }
    }

    private fun initializeCCPAStringTheFirstTime(settings: LegacyExtendedSettings) {
        if (settings.ccpa?.isActive == true && variant != UsercentricsVariant.CCPA) {
            ccpaStrategy.setNotApplicable()
        }
    }

    private fun shouldAcceptAllImplicitlyOnInit(variant: UsercentricsVariant, settings: LegacyExtendedSettings, isInEU: Boolean): Boolean {
        if (this.noShowFlag) {
            return true
        }

        return when (variant) {
            UsercentricsVariant.CCPA -> ccpaStrategy.shouldAcceptAllImplicitlyOnInit()
            UsercentricsVariant.TCF -> tcfStrategy.shouldAcceptAllImplicitlyOnInit(tcf.getGdprAppliesOnTCF())
            UsercentricsVariant.DEFAULT -> gdprStrategy.shouldAcceptAllImplicitlyOnInit(gdprOptions = settings.gdpr, isInEU = isInEU)
        }
    }

    private fun acceptAllImplicitly(controllerId: String) {
        acceptAllImplicitly(controllerId, settingsLegacy.getSettings().services)
    }

    private fun acceptAllImplicitly(controllerId: String, services: List<LegacyService>) {
        services.forEach {
            it.consent = LegacyConsent(status = true, history = it.consent.history)
        }

        dataFacade.execute(
            controllerId = controllerId,
            services = services,
            consentAction = UsercentricsConsentAction.NON_EU_REGION,
            consentType = UsercentricsConsentType.IMPLICIT
        )

        if (settingsLegacy.isTCFEnabled()) {
            tcf.updateIABTCFKeys(tcString = "")

            if (settingsLegacy.isAdditionalConsentModeEnabled()) {
                additionalConsentModeService.acceptAll()
            }
        }

        logAcceptAllImplicitly()
    }

    private fun logAcceptAllImplicitly() {
        val framework = settingsLegacy.getSettings().framework

        val message = when (variant) {
            UsercentricsVariant.CCPA -> formatUSFrameworkMessage(AcceptAllImplicitlyReasons.firstInitializationUSFrameworks, framework)
            UsercentricsVariant.TCF -> AcceptAllImplicitlyReasons.firstInitializationTCF
            UsercentricsVariant.DEFAULT -> AcceptAllImplicitlyReasons.firstInitializationGDPR
            else -> ""
        }
        logger.debug(message)
    }

    private fun denyAllImplicitly(controllerId: String) {
        val services = settingsLegacy.getSettings().services
        services.forEach {
            it.consent = LegacyConsent(status = it.isEssential || (it.defaultConsentStatus ?: false), history = it.consent.history)
        }

        dataFacade.execute(
            controllerId = controllerId,
            services = services,
            consentAction = UsercentricsConsentAction.INITIAL_PAGE_LOAD,
            consentType = UsercentricsConsentType.IMPLICIT,
        )

        if (settingsLegacy.isTCFEnabled()) {
            tcf.updateIABTCFKeys(tcString = "")

            if (settingsLegacy.isAdditionalConsentModeEnabled()) {
                additionalConsentModeService.denyAll()
            }
        }
    }

    private fun mergeSettingsFromStorage(controllerId: String, shouldAcceptAllImplicitlyOnInit: Boolean): MergedServicesSettings? {
        return dataFacade.mergeSettingsFromStorage(controllerId, shouldAcceptAllImplicitlyOnInit)
    }

    override fun resolveInitialView(): InitialView {
        if (this.noShowFlag) {
            return InitialView.NONE
        }

        val variantValue = variant ?: throw IllegalStateException("No variant value")
        val locationValue by lazy { locationService.location }

        val settings = settingsLegacy.getSettings()

        val manualResurfaceTimestamp = settings.renewConsentsTimestampInSeconds?.let { timestamp ->
            convertToManualResurfaceTimestamp(timestamp)
        }

        val versionChangeRequiresReshow = resolveReshow(
            lastInteractionTimestamp = deviceStorage.lastInteractionTimestamp(),
            shouldReshowAfterVersionUpgrade = deviceStorage.getUserActionRequired()
        )
        val shouldManualResurface = shouldManualResurface(manualResurfaceTimestamp)

        val sharedInitialViewOptions = SharedInitialViewOptions(
            versionChangeRequiresReshow = versionChangeRequiresReshow,
            manualResurface = shouldManualResurface
        )

        return when (variantValue) {
            UsercentricsVariant.CCPA -> {
                val ccpaInitialViewOptions = CCPAInitialViewOptions(
                    ccpaOptions = settings.ccpa,
                    framework = settings.framework,
                    sharedInitialViewOptions = sharedInitialViewOptions
                )

                ccpaStrategy.getInitialView(ccpaInitialViewOptions)
            }

            UsercentricsVariant.TCF -> {
                val tcfInitialViewOptions = TCFInitialViewOptions(
                    resurfacePurposeChanged = tcf.getResurfacePurposeChanged(),
                    resurfaceVendorAdded = tcf.getResurfaceVendorAdded(),
                    noGDPRConsentActionPerformed = gdprStrategy.noGDPRConsentActionPerformed(),
                    resurfacePeriodEnded = tcf.getResurfacePeriodEnded(),
                    settingsTCFPolicyVersion = tcf.getSettingsTCFPolicyVersion(),
                    storedTcStringPolicyVersion = tcf.getStoredTcStringPolicyVersion(),
                    resurfaceATPChanged = tcf.getResurfaceATPChanged(),
                    sharedInitialViewOptions = sharedInitialViewOptions
                )

                tcfStrategy.getInitialView(tcfInitialViewOptions)
            }

            UsercentricsVariant.DEFAULT -> {
                val gdprInitialViewOptions = GDPRInitialViewOptions(
                    gdprOptions = settings.gdpr,
                    isInEU = locationValue.isInEU(),
                    sharedInitialViewOptions = sharedInitialViewOptions
                )

                gdprStrategy.getInitialView(gdprInitialViewOptions)
            }
        }
    }

    /**
     * The `isAfterManualResurface` check is necessary because relying solely on `wasLastInteractionBeforeResurface`
     * can lead to incorrect resurface behavior. For example:
     *
     * - Current time: 09:30 AM
     * - Last interaction: 09:00 AM
     * - Resurface time: 10:00 AM
     *
     * In this scenario, the conditions for manual resurface would be met, causing a resurface,
     * and the last interaction timestamp would be updated to 09:30 AM.
     *
     * However, if the current time advances to 09:40 AM, the manual resurface condition would be
     * true again, leading to repeated resurfaces before the intended resurface time.
     *
     * Therefore, we ensure manual resurface only happens when both:
     * 1. The current time is later than the resurface time (`isAfterManualResurface`).
     * 2. The last interaction timestamp is earlier than the resurface time (`wasLastInteractionBeforeResurface`).
     */
    private fun shouldManualResurface(manualResurfaceTimestamp: Long?): Boolean {
        if (manualResurfaceTimestamp == null) return false

        val lastInteractionTimestamp = deviceStorage.lastInteractionTimestamp() ?: 0

        val isAfterManualResurface = (DateTime().timestamp() > manualResurfaceTimestamp)
        val wasLastInteractionBeforeResurface = (lastInteractionTimestamp < manualResurfaceTimestamp)

        return isAfterManualResurface && wasLastInteractionBeforeResurface
    }

    private fun convertToManualResurfaceTimestamp(timestamp: Long) : Long {
        return timestamp * 1000
    }

    private fun resolveReshow(lastInteractionTimestamp: Long?, shouldReshowAfterVersionUpgrade: Boolean): Boolean {
        if (lastInteractionTimestamp == null) {
            return false
        }
        if (shouldReshowAfterVersionUpgrade) {
            return true
        }
        return false
    }
}
