package com.usercentrics.sdk.services.initialValues.variants

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.common.InitialView
import com.usercentrics.sdk.services.ccpa.ICcpa
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.v2.settings.data.CCPASettings

internal interface CCPAStrategy {
    fun shouldAcceptAllImplicitlyOnInit(): Boolean

    fun setNotApplicable()

    fun getInitialView(ccpaInitialViewOptions: CCPAInitialViewOptions, ): InitialView
}

internal class CCPAStrategyImpl(
    private val logger: UsercentricsLogger,
    private val deviceStorage: DeviceStorage,
    private val ccpa: ICcpa,
) : CCPAStrategy {

    override fun shouldAcceptAllImplicitlyOnInit(): Boolean {
        return true
    }

    override fun setNotApplicable() {
        ccpa.setNotApplicable()
    }

    override fun getInitialView(ccpaInitialViewOptions: CCPAInitialViewOptions): InitialView {
        val isOptedOut = ccpa.getCCPAData().optedOut
        val lastTimestamp = deviceStorage.getCcpaTimestampInMillis()
        val isFirstInitialization = lastTimestamp == null
        val showOnFirstTimeVisit = ccpaInitialViewOptions.ccpaOptions?.showOnPageLoad ?: false

        val sharedInitialViewOptions = ccpaInitialViewOptions.sharedInitialViewOptions

        return when {
           sharedInitialViewOptions.manualResurface -> {
                logger.debug(ShowCMPReasons.manualResurface)
                InitialView.FIRST_LAYER
            }

           sharedInitialViewOptions.versionChangeRequiresReshow -> {
                logger.debug(ShowCMPReasons.versionChangeRequiresReshow)
                InitialView.FIRST_LAYER
            }

            isOptedOut == false -> {
                InitialView.NONE
            }

            isFirstInitialization && showOnFirstTimeVisit -> {
                logger.debug(formatUSFrameworkMessage(message = ShowCMPReasons.firstInitUSFramework, framework = ccpaInitialViewOptions.framework))
                InitialView.FIRST_LAYER
            }

            timeToReshowHasPassed(ccpaInitialViewOptions.ccpaOptions, lastTimestamp) -> {
                logger.debug(formatUSFrameworkMessage(message = ShowCMPReasons.shouldReshowAfterTimeUSFramework, framework = ccpaInitialViewOptions.framework))
                InitialView.FIRST_LAYER
            }

            else -> {
                InitialView.NONE
            }
        }
    }

    private fun timeToReshowHasPassed(ccpaOptions: CCPASettings?, lastTimestamp: Long?): Boolean {
        val reshowCmpInDays = ccpaOptions?.reshowAfterDays
        if (reshowCmpInDays == null || lastTimestamp == null) {
            return false
        }

        val sessionDateTime = DateTime(lastTimestamp)
        val elapsedDays = DateTime().day - sessionDateTime.day

        return elapsedDays > reshowCmpInDays
    }
}
