package com.usercentrics.sdk.services.initialValues.variants

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.common.InitialView
import com.usercentrics.sdk.models.settings.GDPROptions
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage

internal interface GDPRStrategy {

    fun shouldAcceptAllImplicitlyOnInit(gdprOptions: GDPROptions?, isInEU: Boolean): Boolean
    fun getInitialView(gdprInitialViewOptions: GDPRInitialViewOptions): InitialView
    fun noGDPRConsentActionPerformed(): Boolean
}

internal class GDPRStrategyImpl(
    private val logger: UsercentricsLogger,
    private val deviceStorage: DeviceStorage
) : GDPRStrategy {

    override fun shouldAcceptAllImplicitlyOnInit(gdprOptions: GDPROptions?, isInEU: Boolean): Boolean {
        val displayCmpOnlyToEUUsers = gdprOptions?.displayCmpOnlyToEUUsers ?: false
        return displayCmpOnlyToEUUsers && !isInEU
    }

    override fun getInitialView(gdprInitialViewOptions: GDPRInitialViewOptions): InitialView {
        val gdprOptions = gdprInitialViewOptions.gdprOptions

        val displayCmpOnlyToEUUsers = gdprOptions?.displayCmpOnlyToEUUsers ?: false
        val reshowBannerInMonths = gdprOptions?.reshowCmpInMonths ?: 0

        val noConsentActionPerformed = noGDPRConsentActionPerformed()

        val sharedInitialViewOptions = gdprInitialViewOptions.sharedInitialViewOptions

        return when {
            sharedInitialViewOptions.manualResurface -> {
                logger.debug(ShowCMPReasons.manualResurface)
                InitialView.FIRST_LAYER
            }

            displayCmpOnlyToEUUsers && !gdprInitialViewOptions.isInEU -> {
                InitialView.NONE
            }

            noConsentActionPerformed -> {
                logger.debug(ShowCMPReasons.noConsentActionPerformedGDPR)
                InitialView.FIRST_LAYER
            }

            timeToReshowHasPassed(reshowBannerInMonths, lastServiceInteractionTimestamp()) -> {
                logger.debug(ShowCMPReasons.shouldReshowAfterTimeGDPR)
                InitialView.FIRST_LAYER
            }

            sharedInitialViewOptions.versionChangeRequiresReshow -> {
                logger.debug(ShowCMPReasons.versionChangeRequiresReshow)
                InitialView.FIRST_LAYER
            }

            else -> {
                InitialView.NONE
            }
        }
    }

    override fun noGDPRConsentActionPerformed(): Boolean {
        return lastServiceInteractionTimestamp() == null
    }

    private fun lastServiceInteractionTimestamp(): Long? {
        return deviceStorage.lastInteractionTimestamp()
    }

    private fun timeToReshowHasPassed(reshowBannerInMonths: Int, lastTimestamp: Long?): Boolean {
        if (reshowBannerInMonths == 0 || lastTimestamp == null) {
            return false
        }

        val lastInteractionPlusReshowBannerThreshold = DateTime(lastTimestamp).addMonths(reshowBannerInMonths)

        val currentTimestamp = DateTime()
        return currentTimestamp > lastInteractionPlusReshowBannerThreshold
    }
}
