package com.usercentrics.sdk.services.settings

import com.usercentrics.sdk.core.settings.SettingsInitializationParameters
import com.usercentrics.sdk.errors.NotInitializedException
import com.usercentrics.sdk.models.settings.LegacyConsent
import com.usercentrics.sdk.models.settings.LegacyExtendedSettings
import com.usercentrics.sdk.v2.settings.facade.ISettingsFacade
import kotlin.coroutines.cancellation.CancellationException

internal class SettingsLegacy(
    private val settingsFacade: ISettingsFacade,
    private val generatorIds: IGeneratorIds
) : ISettingsLegacy {

    private var settings: LegacyExtendedSettings = LegacyExtendedSettings()

    override fun setSettings(settings: LegacyExtendedSettings) {
        this.settings = settings
    }

    override fun setControllerId(controllerId: String) {
        setSettings(settings.copy(controllerId = controllerId))
    }

    override fun getSettings(): LegacyExtendedSettings {
        return this.settings
    }

    override suspend fun initSettings(settingsInitParameters: SettingsInitializationParameters): Result<Unit> {
        return try {
            settingsFacade.loadSettings(settingsInitParameters)
                .fold(
                    onSuccess = { legacyExtendedSettings ->
                        setSettings(legacyExtendedSettings.copy())
                        Result.success(Unit)
                    },
                    onFailure = { error ->
                        Result.failure(NotInitializedException(error.message.toString(), error))
                    }
                )
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            Result.failure(NotInitializedException(e.message.toString(), e))
        }
    }

    override fun isTCFEnabled(): Boolean {
        return this.settings.isTcfEnabled
    }

    override fun isAdditionalConsentModeEnabled(): Boolean {
        return this.settings.tcfui?.isAdditionalConsentModeEnabled ?: false
    }

    override fun selectedAdTechProviders(): List<Int> {
        return this.settings.tcfui?.selectedAdTechProvidersIds ?: emptyList()
    }

    override fun isCCPAEnabled(): Boolean {
        return this.settings.ccpa?.isActive ?: false
    }

    override fun getCCPAIABAgreementExists(): Boolean? {
        return this.settings.ccpa?.iabAgreementExists
    }

    override fun clearConsents() {
        this.settings.apply {
            controllerId = generatorIds.generateControllerId()
            services.forEach {
                it.consent = LegacyConsent(history = listOf(), status = false)
            }
        }

    }
}
