package com.usercentrics.sdk.services.settings

import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.api.ApiSettingsVersion
import com.usercentrics.sdk.models.gdpr.DefaultLabels
import com.usercentrics.sdk.models.gdpr.DefaultUISettings
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.models.tcf.TCFLabels
import com.usercentrics.sdk.models.tcf.TCFUISettings
import com.usercentrics.sdk.ui.color.Color
import com.usercentrics.sdk.ui.color.PredefinedUIColorMachine
import com.usercentrics.sdk.v2.settings.data.UsercentricsService
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization

internal class SettingsMapper(
    private val logger: UsercentricsLogger,
    private val servicesMapper: ISettingsServiceMapper,
    private val generatorIds: IGeneratorIds
) : ISettingsMapper {

    override fun map(
        apiSettings: UsercentricsSettings,
        apiServices: List<UsercentricsService>,
        translations: LegalBasisLocalization,
        controllerId: String?
    ): LegacyExtendedSettings {
        return LegacyExtendedSettings(
            categories = apiSettings.categories ?: emptyList(),
            services = servicesMapper.map(apiSettings, apiServices, translations),
            gdpr = GDPROptions(displayCmpOnlyToEUUsers = apiSettings.displayOnlyForEU, reshowCmpInMonths = apiSettings.reshowBanner),
            ccpa = apiSettings.ccpa,
            controllerId = getValidControllerId(controllerId),
            id = apiSettings.settingsId,
            isTcfEnabled = apiSettings.tcf2Enabled,
            showFirstLayerOnVersionChange = mapShowFirstLayerOnVersionChange(apiSettings.showInitialViewForVersionChange),
            tcfui = mapLegacyTCFUISettings(translations, apiSettings),
            ui = mapUISettings(apiSettings, translations),
            version = apiSettings.version,
            framework = apiSettings.framework?.name,
            renewConsentsTimestampInSeconds = apiSettings.renewConsentsTimestamp
        )
    }

    private fun getValidControllerId(controllerId: String?): String {
        return if (controllerId.isNullOrBlank()) {
            generatorIds.generateControllerId()
        } else {
            controllerId
        }
    }

    private fun mapUISettings(apiSettings: UsercentricsSettings, translations: LegalBasisLocalization): DefaultUISettings? {
        return if (apiSettings.tcf2Enabled) {
            null
        } else {
            DefaultUISettings(
                customization = mapCustomization(apiSettings),
                labels = mapUILabelsFromApiSettings(apiSettings, translations),
                language = mapLanguage(apiSettings),
            )
        }
    }

    private fun mapLegacyTCFUISettings(
        translations: LegalBasisLocalization,
        apiSettings: UsercentricsSettings
    ): TCFUISettings? {
        return if (apiSettings.tcf2Enabled && apiSettings.tcf2 != null) {
            TCFUISettings(
                customization = mapCustomization(apiSettings),
                labels = TCFLabels(
                    vendorsOutsideEU = translations.labels.vendorsOutsideEU,
                    nonTCFLabels = mapUILabelsFromApiSettings(apiSettings, translations),
                    cookieInformation = PredefinedUICookieInformationLabels(
                        anyDomain = apiSettings.labels.anyDomain,
                        day = apiSettings.labels.day,
                        days = apiSettings.labels.days,
                        domain = apiSettings.labels.domain,
                        duration = apiSettings.labels.duration,
                        error = apiSettings.labels.informationLoadingNotPossible,
                        hour = apiSettings.labels.hour,
                        hours = apiSettings.labels.hours,
                        identifier = apiSettings.labels.identifier,
                        loading = apiSettings.labels.loadingStorageInformation,
                        maximumAge = apiSettings.labels.maximumAgeCookieStorage,
                        minute = apiSettings.labels.minute,
                        minutes = apiSettings.labels.minutes,
                        month = apiSettings.labels.month,
                        months = apiSettings.labels.months,
                        multipleDomains = apiSettings.labels.multipleDomains,
                        no = apiSettings.labels.no,
                        nonCookieStorage = apiSettings.labels.nonCookieStorage,
                        second = apiSettings.labels.second,
                        seconds = apiSettings.labels.seconds,
                        session = apiSettings.labels.session,
                        title = apiSettings.labels.storageInformation,
                        titleDetailed = apiSettings.labels.detailedStorageInformation,
                        tryAgain = apiSettings.labels.tryAgain,
                        type = apiSettings.labels.type,
                        year = apiSettings.labels.year,
                        years = apiSettings.labels.years,
                        yes = apiSettings.labels.yes,
                        storageInformationDescription = apiSettings.labels.storageInformationDescription,
                        cookieStorage = translations.labels.cookieStorage,
                        cookieRefresh = translations.labels.cookieRefresh,
                        purposes = apiSettings.tcf2.labelsPurposes
                    )
                ),
                language = mapLanguage(apiSettings),
                isAdditionalConsentModeEnabled = apiSettings.tcf2.isAdditionalConsentModeEnabled(),
                selectedAdTechProvidersIds = apiSettings.tcf2.selectedATPIds,
            )
        } else {
            null
        }
    }

    private fun validateRawColor(name: String, rawValue: String, defaultValue: String): String {
        if (Color.isValid(rawValue)) {
            return rawValue
        }
        logger.warning("Configured color ($rawValue) for '$name' should be a valid hexadecimal, default color will be used $defaultValue")
        return defaultValue
    }

    private fun mapCustomization(apiSettings: UsercentricsSettings): PredefinedUICustomization {
        val customization = apiSettings.customization
        val customizationColor = customization?.color

        val rawLegacyPrimaryColor = customizationColor?.primary.orEmpty()
        val legacyPrimaryColor = validateRawColor("primary", rawLegacyPrimaryColor, PredefinedUICustomizationColor.defaultLegacyPrimary)
        val legacyPrimaryShadedColor = PredefinedUIColorMachine.generateShadedColor(legacyPrimaryColor)

        val rawTextColor = customizationColor?.text.orEmpty()
        val textColor = validateRawColor("text", rawTextColor, PredefinedUICustomizationColor.defaultText)
        val textShadedColor = PredefinedUIColorMachine.generateShadedColor(textColor)

        val rawOverlayColor = customizationColor?.overlay.orEmpty()
        val baseOverlayColor = validateRawColor("overlay", rawOverlayColor, PredefinedUICustomizationColor.defaultOverlayColor)

        val buttonCornerRadius = customization?.borderRadiusButton ?: PredefinedUICustomizationColor.defaultButtonCornerRadius
        val overlayOpacity = runCatching { customization?.overlayOpacity?.toDouble() }.getOrNull() ?: 1.0

        val rawLayerBackgroundColor = customizationColor?.layerBackground.orEmpty()
        val layerBackgroundColor = validateRawColor("layerBackground", rawLayerBackgroundColor, PredefinedUICustomizationColor.defaultBackgroundColor)
        val layerBackgroundLightenColor = PredefinedUIColorMachine.lightenColor(layerBackgroundColor, 0.05)

        val rawTabsBorderColor = customizationColor?.tabsBorderColor.orEmpty()
        val tabsBorderColor = validateRawColor("tabsBorderColor", rawTabsBorderColor, PredefinedUICustomizationColor.defaultTabsBorderColor)

        return PredefinedUICustomization(
            color = PredefinedUICustomizationColor(
                text = textShadedColor,
                acceptAllButton = PredefinedUICustomizationColorButton(
                    background = customizationColor?.acceptBtnBackground.emptyToNull() ?: PredefinedUICustomizationColor.defaultAcceptAllButtonBackground,
                    text = customizationColor?.acceptBtnText?.emptyToNull() ?: PredefinedUICustomizationColor.defaultAcceptAllButtonText,
                    cornerRadius = buttonCornerRadius,
                ),
                denyAllButton = PredefinedUICustomizationColorButton(
                    background = customizationColor?.denyBtnBackground.emptyToNull() ?: PredefinedUICustomizationColor.defaultDenyAllButtonBackground,
                    text = customizationColor?.denyBtnText.emptyToNull() ?: PredefinedUICustomizationColor.defaultDenyAllButtonText,
                    cornerRadius = buttonCornerRadius,
                ),
                saveButton = PredefinedUICustomizationColorButton(
                    background = customizationColor?.saveBtnBackground.emptyToNull() ?: PredefinedUICustomizationColor.defaultSaveButtonBackground,
                    text = customizationColor?.saveBtnText.emptyToNull() ?: PredefinedUICustomizationColor.defaultSaveButtonText,
                    cornerRadius = buttonCornerRadius,
                ),
                manageButton = PredefinedUICustomizationColorButton(
                    background = customizationColor?.saveBtnBackground.emptyToNull() ?: PredefinedUICustomizationColor.defaultSaveButtonBackground,
                    text = customizationColor?.saveBtnText.emptyToNull() ?: PredefinedUICustomizationColor.defaultSaveButtonText,
                    cornerRadius = buttonCornerRadius,
                ),
                okButton = PredefinedUICustomizationColorButton(
                    background = customizationColor?.ccpaButtonColor.emptyToNull() ?: PredefinedUICustomizationColor.defaultOkButtonBackground,
                    text = customizationColor?.ccpaButtonTextColor?.emptyToNull() ?: PredefinedUICustomizationColor.defaultOkButtonText,
                    cornerRadius = buttonCornerRadius,
                ),
                toggles = PredefinedUICustomizationColorToggles(
                    activeBackground = customizationColor?.toggleActiveBackground.emptyToNull() ?: legacyPrimaryShadedColor.color80,
                    inactiveBackground = customizationColor?.toggleInactiveBackground.emptyToNull() ?: textShadedColor.color80,
                    disabledBackground = customizationColor?.toggleDisabledBackground.emptyToNull() ?: textShadedColor.color16,
                    activeIcon = customizationColor?.toggleActiveIcon.emptyToNull() ?: PredefinedUICustomizationColor.defaultToggleIconColor,
                    disabledIcon = customizationColor?.toggleDisabledIcon.emptyToNull() ?: PredefinedUICustomizationColor.defaultToggleIconColor,
                    inactiveIcon = customizationColor?.toggleInactiveIcon.emptyToNull() ?: PredefinedUICustomizationColor.defaultToggleIconColor,
                ),
                layerBackgroundColor = layerBackgroundColor,
                layerBackgroundSecondaryColor = layerBackgroundLightenColor,
                linkColor = customizationColor?.linkFont.emptyToNull() ?: PredefinedUICustomizationColor.defaultLinkColor,
                tabColor = customizationColor?.secondLayerTab.emptyToNull() ?: legacyPrimaryColor,
                baseOverlayColor = baseOverlayColor,
                overlayOpacity = overlayOpacity,
                tabsBorderColor = tabsBorderColor,
            ),
            font = PredefinedUICustomizationFont(
                family = customization?.font?.family.emptyToNull() ?: PredefinedUICustomizationFont.defaultFamily,
                size = customization?.font?.size ?: PredefinedUICustomizationFont.defaultSize
            ),
            logoUrl = customization?.logoUrl,
            cornerRadius = customization?.borderRadiusLayer ?: 8
        )
    }

    private fun mapShowFirstLayerOnVersionChange(showInitialViewForVersionChange: List<String>): List<Int> {
        return showInitialViewForVersionChange.map { element ->
            run {
                when (element) {
                    ApiSettingsVersion.MAJOR.name.lowercase() -> SettingsVersion.MAJOR.ordinal
                    ApiSettingsVersion.MINOR.name.lowercase() -> SettingsVersion.MINOR.ordinal
                    ApiSettingsVersion.PATCH.name.lowercase() -> SettingsVersion.PATCH.ordinal
                    else -> -1
                }
            }
        }
    }

    private fun mapLanguage(apiSettings: UsercentricsSettings): PredefinedUILanguageSettings {
        return PredefinedUILanguageSettings(
            available = apiSettings.languagesAvailable
                .map { PredefinedUILanguage(it) }
                .sortedBy { it.fullName },
            selected = PredefinedUILanguage(apiSettings.language)
        )
    }

    private fun mapUILabelsFromApiSettings(apiSettings: UsercentricsSettings, translations: LegalBasisLocalization): DefaultLabels {
        return DefaultLabels(
            general = PredefinedUIGeneralLabels(
                controllerId = translations.labels.controllerIdTitle,
                date = apiSettings.labels.date,
                decision = apiSettings.labels.decision,
                readMore = apiSettings.labels.btnBannerReadMore,
                more = apiSettings.labels.btnMore,
                acceptAll = apiSettings.labels.btnAcceptAll,
                denyAll = apiSettings.labels.btnDeny,
                continueWithoutAccepting = translations.labels.cnilDenyLinkText
            ),
            service = PredefinedUIServiceLabels(
                dataCollected = PredefinedUIDescriptionTitle(
                    title = apiSettings.labels.dataCollectedList,
                    titleDescription = apiSettings.labels.dataCollectedInfo
                ),
                dataDistribution = PredefinedUIDataDistributionTitle(
                    processingLocationTitle = apiSettings.labels.locationOfProcessing,
                    thirdPartyCountriesTitle = apiSettings.labels.transferToThirdCountries,
                    thirdPartyCountriesDescription = apiSettings.labels.transferToThirdCountriesInfo,
                ),
                dataPurposes = PredefinedUIDescriptionTitle(
                    title = apiSettings.labels.dataPurposes,
                    titleDescription = apiSettings.labels.dataPurposesInfo
                ),
                dataRecipientsTitle = apiSettings.labels.dataRecipientsList,
                descriptionTitle = apiSettings.labels.descriptionOfService,
                history = PredefinedUIDescriptionTitle(
                    title = apiSettings.labels.history,
                    titleDescription = apiSettings.labels.historyDescription
                ),
                legalBasis = PredefinedUIDescriptionTitle(
                    title = apiSettings.labels.legalBasisList,
                    titleDescription = apiSettings.labels.legalBasisInfo
                ),
                processingCompanyTitle = apiSettings.labels.processingCompanyTitle,
                retentionPeriodTitle = apiSettings.labels.retentionPeriod,
                technologiesUsed = PredefinedUIDescriptionTitle(
                    title = apiSettings.labels.technologiesUsed,
                    titleDescription = apiSettings.labels.technologiesUsedInfo
                ),
                urls = PredefinedUIURLsTitle(
                    cookiePolicyTitle = apiSettings.labels.cookiePolicyInfo,
                    dataProcessingAgreementTitle = apiSettings.labels.linkToDpaInfo,
                    optOutTitle = apiSettings.labels.optOut,
                    privacyPolicyTitle = apiSettings.labels.policyOf
                ),
            ),
            ariaLabels = PredefinedUIAriaLabels(
                acceptAllButton = translations.labelsAria?.acceptAllButton,
                ccpaButton = translations.labelsAria?.ccpaButton,
                ccpaMoreInformation = translations.labelsAria?.ccpaMoreInformation,
                closeButton = translations.labelsAria?.closeButton,
                collapse = translations.labelsAria?.collapse,
                cookiePolicyButton = translations.labelsAria?.cookiePolicyButton,
                copyControllerId = translations.labelsAria?.copyControllerId,
                denyAllButton = translations.labelsAria?.denyAllButton,
                expand = translations.labelsAria?.expand,
                fullscreenButton = translations.labelsAria?.fullscreenButton,
                imprintButton = translations.labelsAria?.imprintButton,
                languageSelector = translations.labelsAria?.languageSelector,
                privacyButton = translations.labelsAria?.privacyButton,
                privacyPolicyButton = translations.labelsAria?.privacyPolicyButton,
                saveButton = translations.labelsAria?.saveButton,
                serviceInCategoryDetails = translations.labelsAria?.serviceInCategoryDetails,
                servicesInCategory = translations.labelsAria?.servicesInCategory,
                tabButton = translations.labelsAria?.tabButton,
                usercentricsCMPButtons = translations.labelsAria?.usercentricsCMPButtons,
                usercentricsCMPContent = translations.labelsAria?.usercentricsCMPContent,
                usercentricsCMPHeader = translations.labelsAria?.usercentricsCMPHeader,
                usercentricsCMPUI = translations.labelsAria?.usercentricsCMPUI,
                usercentricsCard = translations.labelsAria?.usercentricsCard,
                usercentricsList = translations.labelsAria?.usercentricsList,
                vendorConsentToggle = translations.labelsAria?.vendorConsentToggle,
                vendorDetailedStorageInformation = translations.labelsAria?.vendorDetailedStorageInformation,
                vendorLegIntToggle = translations.labelsAria?.vendorLegIntToggle,
                logoAltTag = apiSettings.customization?.logoAltTag,
            ),
        )
    }

    private fun String?.emptyToNull(): String? {
        return if (this?.isEmpty() == true) {
            null
        } else {
            this
        }
    }
}
