package com.usercentrics.sdk.services.tcf.interfaces

import com.usercentrics.sdk.v2.settings.data.ConsentDisclosureObject
import com.usercentrics.tcf.core.model.RestrictionType
import com.usercentrics.tcf.core.model.gvl.DataRetention
import com.usercentrics.tcf.core.model.gvl.VendorUrl
import kotlinx.serialization.Contextual
import kotlinx.serialization.Serializable

/** A TCF feature that needs to be disclosed to the end-user. */
@Serializable
data class TCFFeature(
    /** A user-friendly short description that needs to be disclosed to the end-user. */
    val purposeDescription: String,
    /** An official legal list of illustrations that needs to be disclosed to the end-user. */
    val illustrations: List<String>,
    /** The unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** The name as defined in the IAB global vendor list. */
    val name: String,
)

/** TCF purpose that needs to be disclosed to the end-user and requires the end-user's consent and legitimate interest consent.*/
@Serializable
data class TCFPurpose(
    /** A user-friendly short description that needs to be disclosed to the end-user. */
    val purposeDescription: String,
    /** An official legal list of illustrations that needs to be disclosed to the end-user. */
    val illustrations: List<String>,
    /** The unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** The name as defined in the IAB global vendor list. */
    val name: String,
    /** A boolean value which indicates whether the user gave or refused consent, or did not take
     * action yet (null) for this TCF model. */
    val consent: Boolean?,
    /** A boolean value which indicates whether the TCF purpose is part of a selected TCF stack.
     * Stacks can be chosen in the admin interface. */
    val isPartOfASelectedStack: Boolean,
    /** A boolean value which indicates whether the user gave or refused legitimate interest, or
     * did not take action yet (null) for this TCF model. */
    val legitimateInterestConsent: Boolean?,
    /** A boolean value which indicates whether the purpose toggle needs to be displayed in the
     * TCF second layer. */
    val showConsentToggle: Boolean,
    /** A boolean value which indicates whether the legitimate interest toggle needs to be
     * displayed in the TCF second layer. */
    val showLegitimateInterestToggle: Boolean,
    /** The id of the selected TCF stack that this purpose is a part of. Is always null if
     * isPartOfASelectedStack is false and always a number otherwise */
    val stackId: Int?,
    /** The number of vendors that use the TCF Purpose*/
    val numberOfVendors: Int?
)

/** A TCF special purpose that needs to be disclosed to the end-user. */
@Serializable
data class TCFSpecialPurpose(
    /** A user-friendly short description that needs to be disclosed to the end-user. */
    val purposeDescription: String,
    /** An official legal list of illustrations that needs to be disclosed to the end-user. */
    val illustrations: List<String>,
    /** The unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** The name as defined in the IAB global vendor list. */
    val name: String,
)

/** A TCF special feature that needs to be disclosed to the end-user and requires the end-user's consent. */
@Serializable
data class TCFSpecialFeature(
    /** A user-friendly short description that needs to be disclosed to the end-user. */
    val purposeDescription: String,
    /** An official legal list of illustrations that needs to be disclosed to the end-user. */
    val illustrations: List<String>,
    /** The unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** The name as defined in the IAB global vendor list. */
    val name: String,
    /** A boolean value which indicates whether the user gave or refused consent, or did not take
     * action yet (null) for this TCF special feature. */
    val consent: Boolean?,
    /** A boolean value which indicates whether the TCF special feature is part of a selected TCF
     * stack. Stacks can be chosen in the admin interface. */
    val isPartOfASelectedStack: Boolean,
    /** The id of the selected TCF stack that this special feature is a part of. Is always null
     * if isPartOfASelectedStack is false and always a number otherwise */
    val stackId: Int?,
    /** A boolean value which indicates whether the special feature toggle needs to be displayed in the TCF second layer. */
    val showConsentToggle: Boolean,
)

/**
 * A TCF stack is a collection of multiple TCF purposes and TCF special features, with one shared description. On the
 * first layer of a TCF UI, stacks may be shown as alternative to the included single purposes and special features.
 */
@Serializable
data class TCFStack(
    /** A user-friendly short description that needs to be disclosed to the end-user. It summarizes
     * all comprising purposes' and special features' descriptions. */
    val description: String,
    /** This TCF stack's unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** This TCF stack's name as defined in the IAB global vendor list. It summarizes all
     * comprising purposes' and special features' names. */
    val name: String,
    /** All the ids of the TCF purposes that are included in this TCF stack. */
    val purposeIds: List<Int>,
    /** All the ids of the TCF special features that are included in this TCF stack */
    val specialFeatureIds: List<Int>,
)

/**
 * A TCF vendor (aka. advertiser, tracking service, third party) that is registered with the IAB global vendor list.
 * A TCF vendor needs to be disclosed to the end-user and requires the end-user's consent and legitimate interest consent.
 */
@Serializable
data class TCFVendor(
    /** A boolean value which indicates whether the user gave or refused consent, or did not take
     * action yet (null) for this TCF model. */
    val consent: Boolean?,
    /** A list of both name and id of all TCF features that this TCF vendor uses as defined in the
     * IAB global vendor list. */
    val features: List<IdAndName>,
    // TODO change description once service scope are enabled
    /** TCF flexible purposes are not supported yet. This list is currently always empty. */
    val flexiblePurposes: List<IdAndName>,
    /** This TCF vendor's unique id as defined in the IAB global vendor list. */
    val id: Int,
    /** A boolean value which indicates whether the user gave or refused legitimate interest, or
     * did not take action yet (null) for this TCF model. */
    val legitimateInterestConsent: Boolean?,
    /** A list of both name and id of all TCF purposes that this TCF vendor uses by legitimate
     * interest as defined in the IAB global vendor list. */
    val legitimateInterestPurposes: List<IdAndName>,
    /** This TCF vendor's name as defined in the IAB global vendor list. */
    val name: String,
    /** This TCF vendor's link to it's privacy policy website as defined in the IAB global vendor list. */
    val policyUrl: String,
    /** A list of both name and id of all TCF purposes that this TCF vendor uses as defined in the
     * IAB global vendor list. */
    val purposes: List<IdAndName>,
    /** A list of all publisher restrictions for this vendors purposes */
    val restrictions: List<TCFVendorRestriction>,
    /** A list of both name and id of all TCF special purposes that this TCF vendor uses as defined
     * in the IAB global vendor list. */
    val specialFeatures: List<IdAndName>,
    /** A list of both name and id of all TCF special purposes that this TCF vendor uses as defined
     * in the IAB global vendor list. */
    val specialPurposes: List<IdAndName>,
    /** A boolean value which indicates whether consent toggle needs to be displayed in the TCF
     * second layer for the vendor. Is true if purposes are processed by consent. */
    val showConsentToggle: Boolean,
    /** A boolean value which indicates whether the legitimate interest toggle needs to be
     * displayed in the TCF second layer. */
    val showLegitimateInterestToggle: Boolean,
    /** The number, in seconds, of the longest potential duration for storage on a device, as set
     * when using the cookie method of storage. A negative number or a 0 indicate session storage
     * similar to the Set-Cookie spec. If a vendor only uses non-cookie storage the value should
     * not be positive. Note: this only includes what is declared when the storage is set and does
     * not consider duration extensions should storage be refreshed. */
    val cookieMaxAgeSeconds: Double? = null,
    /** Indicates the vendor’s use of non-cookie storage and access to information already stored
     * on a user’s device. True indicates non-cookie access is used. False indicates only cookie
     * storage and access are used. */
    val usesNonCookieAccess: Boolean,
    /** Location of vendor-hosted deviceStorage.json file */
    val deviceStorageDisclosureUrl: String? = null,
    /** Indicates whether the vendor uses cookie storage (session or otherwise). */
    val usesCookies: Boolean = false,
    /** Indicates the vendor’s refreshing a cookie */
    val cookieRefresh: Boolean? = false,
    /** Indicates the vendor’s is sharing data outside the European Union */
    val dataSharedOutsideEU: Boolean? = false,
    /** Indicates the retention period that vendor's will keep for purposes */
    val dataRetention: DataRetention? = null,
    /** A list of both name and id of all TCF data categories that this TCF vendor uses as defined in the IAB global vendor list. */
    val dataCategories: List<IdAndName>,
    /** A list that contains the urls that this TCF vendor uses for privacy and legitimate interest */
    val vendorUrls: List<VendorUrl>
)

/** A TCF vendor restriction for a flexible purpose */
@Serializable
data class TCFVendorRestriction(
    /** The purpose id */
    val purposeId: Int,
    /** The restriction type */
    @Contextual val restrictionType: RestrictionType,
)

/** All the data that needs to be disclosed to the end-user if TCF is enabled. */
@Serializable
data class TCFData(
    /** A list of all the TCF features that need to be disclosed to the end-user if TCF is enabled */
    val features: List<TCFFeature>,
    /** A list of all the TCF purposes that need to be disclosed to the end-user if TCF is enabled */
    val purposes: List<TCFPurpose>,
    /** A list of all the TCF special features that need to be disclosed to the end-user if TCF is enabled */
    val specialFeatures: List<TCFSpecialFeature>,
    /** A list of all the TCF special purposes that need to be disclosed to the end-user if TCF is enabled */
    val specialPurposes: List<TCFSpecialPurpose>,
    /**
     * A list of all TCF stacks that need to be disclosed to the end-user if TCF is enabled. On the first layer of a
     * TCF UI, stacks may be shown as alternative its comprising purposes and special features. However, on the second
     * layer, this is not allowed according to the IAB TCF v.2.0 guidelines. Also on the first layer, a purpose or special
     * feature may not be shown if a stack containing this purpose or special feature is already displayed. Note, that all
     * purposes and special features that are part of stacks will still be listed in the TCFData purposes and special
     * features fields, but they will be flagged with isPartOfASelectedStack = true and include a non-null stackId.
     */
    val stacks: List<TCFStack>,
    /** A list of all TCF vendors that need to be disclosed to the end-user if TCF is enabled */
    val vendors: List<TCFVendor>,
    /** The TCF string */
    val tcString: String,
    /** The total of vendors and services */
    val thirdPartyCount: Int,
)

@Serializable
data class IdAndName(
    val id: Int,
    val name: String,
)

@Serializable
data class IdAndConsent(
    val id: Int,
    val consent: Boolean?,
    val legitimateInterestConsent: Boolean?,
)
