package com.usercentrics.sdk.v2.async.dispatcher

import kotlinx.coroutines.*

@OptIn(ExperimentalCoroutinesApi::class)
internal open class Dispatcher(
    mainDispatcher: CoroutineDispatcher,
    asyncDispatcher: CoroutineDispatcher
) {

    private val mainDispatcher: CoroutineDispatcher
    private val asyncDispatcher: CoroutineDispatcher

    init {
        this.mainDispatcher = mainDispatcher.limitedParallelism(1)
        this.asyncDispatcher = asyncDispatcher.limitedParallelism(1)
    }

    fun <T> dispatch(block: suspend DispatcherScope.() -> T): DispatcherCallback<T> {
        val dispatcherCallback = DispatcherCallback<T>()

        asyncDispatcher.scope().launch {
            runAsyncScope(block, dispatcherCallback)
        }
        return dispatcherCallback
    }

    fun dispatchMain(block: () -> Unit) {
        mainDispatcher.scope().launch { block() }
    }

    fun <T> dispatchWithTimeout(timeout: Long, block: suspend DispatcherScope.() -> T): DispatcherCallback<T> {
        val dispatcherCallback = DispatcherCallback<T>()

        asyncDispatcher.scope().launch {
            try {
                withTimeoutOrNull<Any?>(timeout) {
                    suspendCancellableCoroutine { continuation ->
                        val job = launch { runAsyncScope(block, dispatcherCallback) }

                        continuation.invokeOnCancellation {
                            job.cancel()
                        }
                    }
                }
            } catch (ex: AssertionError) {
                throw ex
            }
        }
        return dispatcherCallback
    }

    private suspend fun <T> runAsyncScope(block: suspend DispatcherScope.() -> T, dispatcherCallback: DispatcherCallback<T>) {
        val dispatcherScope = DispatcherScope(asyncDispatcher)
        val resultBlock = runCatching { block(dispatcherScope) }
        rethrowAssertion(resultBlock.exceptionOrNull())
        dispatcherCallback.setResult(resultBlock)
    }

    private fun rethrowAssertion(cause: Throwable?) {
        if (cause is AssertionError) {
            throw cause
        }
    }
}

internal fun CoroutineDispatcher.scope(): CoroutineScope {
    return CoroutineScope(SupervisorJob() + this)
}

