package com.usercentrics.sdk.v2.banner.service

import com.usercentrics.sdk.acm.service.AdditionalConsentModeService
import com.usercentrics.sdk.models.common.UsercentricsVariant
import com.usercentrics.sdk.models.settings.LegacyExtendedSettings
import com.usercentrics.sdk.models.settings.PredefinedTVViewSettings
import com.usercentrics.sdk.services.ccpa.ICcpa
import com.usercentrics.sdk.services.settings.ISettingsLegacy
import com.usercentrics.sdk.services.tcf.TCFUseCase
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import com.usercentrics.sdk.v2.banner.model.PredefinedUIViewData
import com.usercentrics.sdk.v2.banner.service.mapper.ccpa.CCPAViewSettingsMapper
import com.usercentrics.sdk.v2.banner.service.mapper.gdpr.GDPRViewSettingsMapper
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.TCFViewSettingsMapper
import com.usercentrics.sdk.v2.settings.service.ISettingsService
import com.usercentrics.sdk.v2.translation.service.ITranslationService

internal class BannerViewDataServiceImpl(
    private val settingsService: ISettingsService,
    private val settingsLegacy: ISettingsLegacy,
    private val translationService: ITranslationService,
    private val tcfInstance: TCFUseCase,
    private val ccpaInstance: ICcpa,
    private val additionalConsentModeService: AdditionalConsentModeService,
    private val variant: UsercentricsVariant,
    private val dispatcher: Dispatcher,
) : BannerViewDataService {

    companion object {
        internal const val DEFAULT_CCPA_TOGGLE_VALUE = false
    }

    override fun buildViewData(callback: (PredefinedUIViewData) -> Unit) {
        val settingsLegacyData = settingsLegacy.getSettings()

        dispatcher.dispatch {
            when (variant) {
                UsercentricsVariant.DEFAULT -> getGDPRMapper(settingsLegacyData).map()
                UsercentricsVariant.CCPA -> getCCPAMapper(settingsLegacyData).map()
                UsercentricsVariant.TCF -> {
                    val tcfData = tcfInstance.getTCFData()
                    getTCFMapper(settingsLegacyData, tcfData).map()
                }
            }
        }.onSuccess {
            dispatcher.dispatchMain {
                callback(
                    PredefinedUIViewData(
                        controllerId = settingsLegacyData.controllerId,
                        uiVariant = variant,
                        settings = it
                    )
                )
            }
        }
    }

    override fun buildTVViewSettings(callback: (PredefinedTVViewSettings) -> Unit) {
        val settingsLegacyData = settingsLegacy.getSettings()

        dispatcher.dispatch {
            when (variant) {
                UsercentricsVariant.DEFAULT -> getGDPRMapper(settingsLegacyData).mapTV()
                UsercentricsVariant.TCF -> {
                    val tcfData = tcfInstance.getTCFData()
                    getTCFMapper(settingsLegacyData, tcfData).mapTV()
                }

                UsercentricsVariant.CCPA -> throw IllegalStateException("CCPA is not supported for TV")
            }
        }.onSuccess {
            dispatcher.dispatchMain {
                callback(it)
            }
        }
    }

    private fun getGDPRMapper(settingsLegacyData: LegacyExtendedSettings): GDPRViewSettingsMapper {
        val settingsState = settingsLegacy.getSettings()
        return GDPRViewSettingsMapper(
            settings = settingsService.settings!!.data,
            customization = settingsLegacyData.ui!!.customization,
            labels = settingsLegacyData.ui.labels,
            translations = translationService.translations!!,
            controllerId = settingsLegacyData.controllerId,
            categories = settingsState.categories,
            services = settingsState.services,
        )
    }

    private fun getCCPAMapper(settingsLegacyData: LegacyExtendedSettings): CCPAViewSettingsMapper {
        val settingsState = settingsLegacy.getSettings()
        return CCPAViewSettingsMapper(
            settings = settingsService.settings!!.data,
            customization = settingsLegacyData.ui!!.customization,
            labels = settingsLegacyData.ui.labels,
            controllerId = settingsLegacyData.controllerId,
            categories = settingsState.categories,
            services = settingsState.services,
            optOutToggleInitialValue = getCCPAOptedOut(),
            translations = translationService.translations!!
        )
    }

    private fun getTCFMapper(settingsLegacyData: LegacyExtendedSettings, tcfData: TCFData): TCFViewSettingsMapper {
        val settingsState = settingsLegacy.getSettings()

        return TCFViewSettingsMapper(
            settings = settingsService.settings!!.data,
            translations = translationService.translations!!,
            tcfData = tcfData,
            customization = settingsLegacyData.tcfui!!.customization,
            categories = settingsState.categories,
            services = settingsState.services,
            labels = settingsLegacyData.tcfui.labels,
            controllerId = settingsLegacyData.controllerId,
            adTechProviders = additionalConsentModeService.adTechProviderList ?: emptyList()
        )
    }

    private fun getCCPAOptedOut(): Boolean {
        return ccpaInstance.getCCPAData().optedOut ?: DEFAULT_CCPA_TOGGLE_VALUE
    }

}
