package com.usercentrics.sdk.v2.banner.service.mapper.gdpr

import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.extensions.emptyToNull
import com.usercentrics.sdk.models.gdpr.DefaultLabels
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.v2.banner.service.mapper.FooterButtonLayoutMapper
import com.usercentrics.sdk.v2.banner.service.mapper.PoweredByMapper
import com.usercentrics.sdk.v2.settings.data.FirstLayerCloseOption
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.SecondLayerTrigger
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings

internal class GDPRFirstLayerMapper(
    private val settings: UsercentricsSettings,
    private val customization: PredefinedUICustomization,
    private val labels: DefaultLabels,
) {

    companion object {
        private val defaultLogoPosition = FirstLayerLogoPosition.LEFT
    }

    fun map(): UCUIFirstLayerSettings {
        return UCUIFirstLayerSettings(
            layout = settings.firstLayer?.mobileVariant ?: UCUIFirstLayerSettings.defaultLayout,
            headerSettings = header(),
            footerSettings = footer(),
            contentSettings = emptyList(),
        )
    }

    fun mapTV(): PredefinedTVFirstLayerSettings {
        val shortDescription = if (settings.bannerMobileDescriptionIsActive) settings.firstLayerMobileDescriptionHtml.emptyToNull() else null
        return PredefinedTVFirstLayerSettings(
            title = settings.labels.firstLayerTitle,
            content = shortDescription ?: settings.firstLayerDescriptionHtml ?: "",
            logoUrl = settings.customization?.logoUrl,
            primaryActions = tvButtons(),
            secondaryActions = tvLinks(),
        )
    }

    private fun header(): PredefinedUIHeaderSettings {
        val shortDescription = if (settings.bannerMobileDescriptionIsActive) {
            settings.firstLayerMobileDescriptionHtml
        } else {
            null
        }

        val closeLink = if (settings.firstLayer?.closeOption == FirstLayerCloseOption.LINK) {
            labels.general.continueWithoutAccepting
        } else {
            null
        }

        return PredefinedUIFirstLayerHeaderSettings(
            shortDescription = shortDescription,
            contentDescription = settings.firstLayerDescriptionHtml ?: "",
            title = settings.labels.firstLayerTitle,
            logoPosition = settings.firstLayer?.logoPosition ?: defaultLogoPosition,
            language = null,
            logoURL = settings.customization?.logoUrl,
            links = headerLinks(),
            firstLayerCloseLink = closeLink.emptyToNull(),
            firstLayerCloseIcon = settings.firstLayer?.closeOption?.equals(FirstLayerCloseOption.ICON),
            readMoreText = settings.labels.btnBannerReadMore
        )
    }

    private fun headerLinks(): List<PredefinedUILink> {
        val privacyLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.privacyPolicyLinkText,
            url = settings.privacyPolicyUrl,
            eventType = UsercentricsAnalyticsEventType.PRIVACY_POLICY_LINK,
        )
        val imprintLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.imprintLinkText,
            url = settings.imprintUrl,
            eventType = UsercentricsAnalyticsEventType.IMPRINT_LINK,
        )
        val moreLink = if (settings.firstLayer?.secondLayerTrigger == SecondLayerTrigger.LINK) {
            PredefinedUILink.moreLink(settings.labels.btnMore)
        } else {
            null
        }
        return listOfNotNull(privacyLink, imprintLink, moreLink).filter { !it.isEmpty() }
    }

    private fun footer(): PredefinedUIFooterSettings {
        val buttonLayoutMapper = footerButtonsMapper()

        return PredefinedUIFooterSettings(
            poweredBy = poweredBy(),
            buttons = buttonLayoutMapper.mapButtons(),
            buttonsLandscape = buttonLayoutMapper.mapButtonsLandscape(),
        )
    }

    private fun poweredBy(): PredefinedUIFooterEntry? {
        return PoweredByMapper.mapPoweredBy(
            LegacyPoweredBy(
                isEnabled = settings.enablePoweredBy,
            )
        )
    }

    private fun footerButtonsMapper(): FooterButtonLayoutMapper {
        val acceptAllButton = PredefinedUIFooterButton(
            label = settings.labels.btnAcceptAll, customization = customization.color.acceptAllButton, type = PredefinedUIButtonType.ACCEPT_ALL
        )
        val manageSettingsButton = if (settings.firstLayer?.secondLayerTrigger == SecondLayerTrigger.BUTTON) {
            PredefinedUIFooterButton(
                label = settings.labels.btnMore, customization = customization.color.manageButton, type = PredefinedUIButtonType.MANAGE_SETTINGS
            )
        } else null

        val denyAllButton: PredefinedUIFooterButton? = if (showDenyAllButton()) {
            PredefinedUIFooterButton(
                label = settings.labels.btnDeny, customization = customization.color.denyAllButton, type = PredefinedUIButtonType.DENY_ALL
            )
        } else {
            null
        }

        return FooterButtonLayoutMapper(
            acceptAll = acceptAllButton, manageSettings = manageSettingsButton, denyAll = denyAllButton
        )
    }

    private fun tvButtons(): List<PredefinedTVActionButton> {
        val denyAllButton = if (showDenyAllButton()) {
            PredefinedTVActionButton(
                label = settings.labels.btnDeny, type = PredefinedTVActionButtonType.DenyAll
            )
        } else {
            null
        }

        return listOfNotNull(
            PredefinedTVActionButton(
                label = settings.labels.btnAcceptAll, type = PredefinedTVActionButtonType.AcceptAll
            ),
            denyAllButton,
            PredefinedTVActionButton(
                label = settings.labels.btnMore, type = PredefinedTVActionButtonType.More()
            ),

            ).filter { it.label.isNotEmpty() }
    }

    private fun tvLinks(): List<PredefinedTVActionButton> {
        return listOfNotNull(settings.privacyPolicyUrl?.let {
            PredefinedTVActionButton(
                label = settings.labels.privacyPolicyLinkText, type = PredefinedTVActionButtonType.Url(it)
            )
        }, settings.imprintUrl?.let {
            PredefinedTVActionButton(
                label = settings.labels.imprintLinkText, type = PredefinedTVActionButtonType.Url(it)
            )
        }).filter { it.label.isNotEmpty() }
    }

    private fun showDenyAllButton(): Boolean {
        return settings.firstLayer?.hideButtonDeny == false
    }
}
