package com.usercentrics.sdk.v2.banner.service.mapper.tcf

import com.usercentrics.sdk.models.settings.PredefinedTVSecondLayerDetailsEntry
import com.usercentrics.sdk.models.tcf.TCFLabels
import com.usercentrics.sdk.services.tcf.interfaces.TCFVendor
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.getVendorUrls
import com.usercentrics.sdk.v2.settings.data.TCF2Settings
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.tcf.core.model.gvl.VendorUrl

internal class TCFDetailsMapperTV(
    private val settings: UsercentricsSettings,
    private val labels: TCFLabels,
) {

    private val tcf2: TCF2Settings
        get() = settings.tcf2!!

    fun map(vendor: TCFVendor): List<PredefinedTVSecondLayerDetailsEntry> {
        return listOfNotNull(
            getPurposesProcessedByConsent(vendor),
            getLegitimateInterestPurposes(vendor),
            getSpecialPurposes(vendor),
            getFeatures(vendor),
            getSpecialFeatures(vendor),
            getCategoriesOfData(vendor),
            getDataSharedOutsideEU(vendor),
            getPolicyURL(vendor),
            getLegitimateInterestURL(vendor),
            getDataRetentionPeriod(vendor)
//            vendor.getStorageInformation(),
        )
    }

    private fun getPurposesProcessedByConsent(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        return if (vendor.purposes.isNotEmpty()) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = tcf2.vendorPurpose,
                content = PredefinedTVSecondLayerDetailsEntry.mapPurposesWithRetention(
                    idAndNameList = vendor.purposes,
                    retentionPeriod = vendor.dataRetention?.purposes,
                    label = tcf2.dataRetentionPeriodLabel
                )
            )
        } else {
            null
        }
    }

    private fun getLegitimateInterestPurposes(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        return if (vendor.legitimateInterestPurposes.isNotEmpty()) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = tcf2.vendorLegitimateInterestPurposes,
                content = PredefinedTVSecondLayerDetailsEntry.mapContent(vendor.legitimateInterestPurposes.map { it.name })
            )
        } else {
            null
        }
    }

    private fun getSpecialPurposes(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        return if (vendor.specialPurposes.isNotEmpty()) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = tcf2.vendorSpecialPurposes,
                content = PredefinedTVSecondLayerDetailsEntry.mapPurposesWithRetention(
                    idAndNameList = vendor.specialPurposes,
                    retentionPeriod = vendor.dataRetention?.specialPurposes,
                    label = tcf2.dataRetentionPeriodLabel
                )
            )
        } else {
            null
        }
    }

    private fun getFeatures(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        return if (vendor.features.isNotEmpty()) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = tcf2.vendorFeatures,
                content = PredefinedTVSecondLayerDetailsEntry.mapContent(vendor.features.map { it.name })
            )
        } else {
            null
        }
    }

    private fun getSpecialFeatures(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        return if (vendor.specialFeatures.isNotEmpty()) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = tcf2.vendorSpecialFeatures,
                content = PredefinedTVSecondLayerDetailsEntry.mapContent(vendor.specialFeatures.map { it.name })
            )
        } else {
            null
        }
    }

    private fun getCategoriesOfData(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        if (vendor.dataCategories.isEmpty()) {
            return null
        }

        return PredefinedTVSecondLayerDetailsEntry.TitleContent(
            title = tcf2.categoriesOfDataLabel,
            content = PredefinedTVSecondLayerDetailsEntry.mapContent(vendor.dataCategories.map { it.name })
        )
    }

    private fun getDataSharedOutsideEU(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        val title = settings.labels.transferToThirdCountries
        val description = labels.vendorsOutsideEU
        val hasInfo = title.isNotBlank() && description.isNotBlank()
        return if (vendor.dataSharedOutsideEU == true && hasInfo) {
            PredefinedTVSecondLayerDetailsEntry.TitleContent(
                title = title,
                content = description
            )
        } else {
            null
        }
    }

    private fun getPolicyURL(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        val policyUrl = getVendorUrls(vendor, settings)?.privacy ?: return null

        return PredefinedTVSecondLayerDetailsEntry.Link(
            title = settings.labels.privacyPolicyLinkText,
            linkLabel = settings.labels.policyOf,
            url = policyUrl,
        )
    }

    private fun getLegitimateInterestURL(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        val legIntClaimUrl = getVendorUrls(vendor, settings)?.legIntClaim ?: return null

        return PredefinedTVSecondLayerDetailsEntry.Link(
            title = tcf2.legitimateInterestLabel,
            linkLabel = settings.labels.policyOf,
            url = legIntClaimUrl,
        )
    }

    private fun getDataRetentionPeriod(vendor: TCFVendor): PredefinedTVSecondLayerDetailsEntry? {
        if (vendor.dataRetention?.stdRetention == null) {
            return null
        }

        return PredefinedTVSecondLayerDetailsEntry.TitleContent(
            title = tcf2.dataRetentionPeriodLabel,
            content = vendor.dataRetention.stdRetention.toString()
        )
    }
}
