package com.usercentrics.sdk.v2.banner.service.mapper.tcf

import com.usercentrics.sdk.StackProps
import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.UsercentricsMaps
import com.usercentrics.sdk.extensions.emptyToNull
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.v2.banner.service.mapper.FooterButtonLayoutMapper
import com.usercentrics.sdk.v2.banner.service.mapper.PoweredByMapper
import com.usercentrics.sdk.v2.settings.data.FirstLayerCloseOption
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings

internal class TCFFirstLayerMapper(
    private val settings: UsercentricsSettings,
    private val tcfData: TCFData,
    private val customization: PredefinedUICustomization,
    private val categories: List<UsercentricsCategory>,
    private val services: List<LegacyService>
) {

    companion object {
        private val defaultLogoPosition = FirstLayerLogoPosition.LEFT
    }

    private val hasToggles: Boolean = !settings.tcf2!!.firstLayerHideToggles
    private val hideLegitimateInterestToggles: Boolean = settings.tcf2!!.hideLegitimateInterestToggles
    private val manageIsLink
        get() = hasToggles

    private val stacks by lazy { UsercentricsMaps.mapStacks(tcfData) }

    fun map() = UCUIFirstLayerSettings(
        layout = settings.tcf2?.firstLayerMobileVariant ?: UCUIFirstLayerSettings.defaultLayout,
        headerSettings = headerSettings(),
        footerSettings = footerSettings(),
        contentSettings = contentSettings()
    )

    fun mapTV(): PredefinedTVFirstLayerSettings {
        return PredefinedTVFirstLayerSettings(
            title = settings.tcf2!!.firstLayerTitle,
            content = contentTv(),
            logoUrl = settings.customization?.logoUrl,
            primaryActions = tvButtons(),
            secondaryActions = tvLinks(),
        )
    }

    private fun headerSettings(): PredefinedUIHeaderSettings {
        return PredefinedUIFirstLayerHeaderSettings(
            title = settings.tcf2!!.firstLayerTitle,
            links = headerLinks(),
            contentDescription = headerMessage(),
            logoPosition = settings.firstLayer?.logoPosition ?: defaultLogoPosition,
            logoURL = settings.customization?.logoUrl,
            firstLayerCloseIcon = settings.firstLayer?.closeOption?.equals(FirstLayerCloseOption.ICON),
            shortDescription = null,
            firstLayerCloseLink = null,
            language = null,
            readMoreText = settings.labels.btnBannerReadMore
        )
    }

    private fun headerMessage(): String {
        val messageBuilder = StringBuilder()
        val vendorsCount = tcfData.thirdPartyCount

        // Main Description
        settings.tcf2!!.firstLayerDescription?.trim()?.emptyToNull()?.let { text ->
            messageBuilder.append(text.replace("%VENDOR_COUNT%", vendorsCount.toString()))
        }

        // Additional Message
        settings.tcf2.firstLayerAdditionalInfo?.trim()?.emptyToNull()?.let { text ->
            if (messageBuilder.isNotEmpty()) {
                messageBuilder.append(" ")
            }
            messageBuilder.append(text)
        }

        // Resurface Message
        settings.tcf2.appLayerNoteResurface?.trim()?.emptyToNull()?.let { text ->
            if (messageBuilder.isNotEmpty()) {
                messageBuilder.append(" ")
            }
            messageBuilder.append(text)
        }

        // Data shared outside EU Message
        val dataSharedOutsideEUText = settings.tcf2.dataSharedOutsideEUText?.trim() ?: ""
        val showDataSharedOutsideEUText = settings.tcf2.showDataSharedOutsideEUText
        if (showDataSharedOutsideEUText && dataSharedOutsideEUText.isNotBlank()) {
            if (messageBuilder.isNotEmpty()) {
                messageBuilder.append("<br><br>")
            }
            messageBuilder.append(dataSharedOutsideEUText)
        }

        return messageBuilder.toString()
    }

    private fun headerLinks(): List<PredefinedUILink> {
        val manageLink = if (manageIsLink) PredefinedUILink.moreLink(
            label = settings.tcf2!!.linksManageSettingsLabel,
        ) else null
        val vendorsLink = PredefinedUILink(
            label = settings.tcf2!!.linksVendorListLinkLabel,
            url = null,
            linkType = PredefinedUILinkType.VENDOR_LIST,
            eventType = UsercentricsAnalyticsEventType.MORE_INFORMATION_LINK,
        )
        val privacyLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.privacyPolicyLinkText,
            url = settings.privacyPolicyUrl,
            eventType = UsercentricsAnalyticsEventType.PRIVACY_POLICY_LINK,
        )
        val imprintLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.imprintLinkText,
            url = settings.imprintUrl,
            eventType = UsercentricsAnalyticsEventType.IMPRINT_LINK,
        )
        return listOfNotNull(privacyLink, imprintLink, manageLink, vendorsLink).filter { !it.isEmpty() }
    }

    private fun footerSettings(): PredefinedUIFooterSettings {
        val manageSettingsButton = if (manageIsLink) {
            null
        } else {
            PredefinedUIFooterButton(
                label = settings.tcf2!!.linksManageSettingsLabel,
                customization = customization.color.manageButton,
                type = PredefinedUIButtonType.MANAGE_SETTINGS,
            )
        }
        val saveSettingsButton = if (hasToggles) {
            PredefinedUIFooterButton(
                label = settings.tcf2!!.buttonsSaveLabel,
                customization = customization.color.saveButton,
                type = PredefinedUIButtonType.SAVE_SETTINGS,
            )
        } else {
            null
        }
        val denyAllButton = if (settings.tcf2?.firstLayerHideButtonDeny == true) {
            null
        } else {
            PredefinedUIFooterButton(
                label = settings.tcf2!!.buttonsDenyAllLabel,
                customization = customization.color.denyAllButton,
                type = PredefinedUIButtonType.DENY_ALL,
            )
        }
        val acceptAllButton = PredefinedUIFooterButton(
            label = settings.tcf2.buttonsAcceptAllLabel,
            customization = customization.color.acceptAllButton,
            type = PredefinedUIButtonType.ACCEPT_ALL,
        )
        val buttonLayoutMapper = FooterButtonLayoutMapper(
            acceptAll = acceptAllButton,
            denyAll = denyAllButton,
            saveSettings = saveSettingsButton,
            manageSettings = manageSettingsButton,
        )
        return PredefinedUIFooterSettings(
            poweredBy = poweredBy(),
            buttons = buttonLayoutMapper.mapButtons(),
            buttonsLandscape = buttonLayoutMapper.mapButtonsLandscape(),
        )
    }

    private fun tvButtons(): List<PredefinedTVActionButton> {
        val denyAllButton = if (settings.tcf2?.firstLayerHideButtonDeny == true) {
            null
        } else {
            PredefinedTVActionButton(
                label = settings.tcf2?.buttonsDenyAllLabel ?: "",
                type = PredefinedTVActionButtonType.DenyAll,
            )
        }

        return listOfNotNull(
            PredefinedTVActionButton(
                label = settings.tcf2?.buttonsAcceptAllLabel ?: "",
                type = PredefinedTVActionButtonType.AcceptAll,
            ),
            denyAllButton,
            PredefinedTVActionButton(
                label = settings.tcf2?.linksManageSettingsLabel ?: "",
                type = PredefinedTVActionButtonType.More(),
            ),
            PredefinedTVActionButton(
                label = settings.tcf2?.linksVendorListLinkLabel ?: "",
                type = PredefinedTVActionButtonType.More(settings.tcf2?.labelsIabVendors),
            ),
        ).filter { it.label.isNotEmpty() }
    }

    private fun tvLinks(): List<PredefinedTVActionButton> {
        return listOfNotNull(settings.privacyPolicyUrl?.let {
            PredefinedTVActionButton(
                label = settings.labels.privacyPolicyLinkText,
                type = PredefinedTVActionButtonType.Url(it),
            )
        }, settings.imprintUrl?.let {
            PredefinedTVActionButton(
                label = settings.labels.imprintLinkText,
                type = PredefinedTVActionButtonType.Url(it),
            )
        }).filter { it.label.isNotEmpty() }
    }

    private fun poweredBy(): PredefinedUIFooterEntry? {
        return PoweredByMapper.mapPoweredBy(LegacyPoweredBy(isEnabled = settings.enablePoweredBy))
    }

    private fun contentTv(): String {
        val messageBuilder = StringBuilder()
        messageBuilder.append(headerMessage())

        fun appendPredefinedUICardUISectionToMessageBuilder(section: PredefinedUICardUISection?) {
            if (section != null) {
                messageBuilder.append("<br><br>")
                messageBuilder.append(
                    "${section.title}: ${section.cards.joinToString { it.title }}"
                )
            }
        }

        appendPredefinedUICardUISectionToMessageBuilder(purposesCardsSection())
        appendPredefinedUICardUISectionToMessageBuilder(specialPurposesCardsSection())
        appendPredefinedUICardUISectionToMessageBuilder(specialFeaturesCardsSection())
        appendPredefinedUICardUISectionToMessageBuilder(nonIABPurposesCardsSection())

        return messageBuilder.toString()
    }

    private fun contentSettings(): List<PredefinedUICardUISection> {
        val cardUISections = mutableListOf<PredefinedUICardUISection>()

        val purposesCardsSection = purposesCardsSection()
        if (purposesCardsSection != null) {
            cardUISections.add(purposesCardsSection)
        }

        val specialPurposeProps = specialPurposesCardsSection()
        if (specialPurposeProps != null) {
            cardUISections.add(specialPurposeProps)
        }

        val specialFeaturesCardsSection = specialFeaturesCardsSection()
        if (specialFeaturesCardsSection != null) {
            cardUISections.add(specialFeaturesCardsSection)
        }

        val nonIABPurposesCardsSection = nonIABPurposesCardsSection()
        if (nonIABPurposesCardsSection != null) {
            cardUISections.add(nonIABPurposesCardsSection)
        }

        return cardUISections
    }

    private fun purposesCardsSection(): PredefinedUICardUISection? {
        if (tcfData.purposes.isEmpty()) {
            return null
        }

        val purposesProps = UsercentricsMaps.mapPurposes(tcfData).map { TCFHolder(it, hasMainToggle = hasToggles, hideLegitimateInterestToggles = hideLegitimateInterestToggles) }
        val stacksProps = stacks.filter { it.stack.purposeIds.isNotEmpty() }.map { mapStackPropsToTCFHolder(it, it.stack.purposeIds, purposesProps) }

        return mapCardsSectionFromTCFHolder(
            sectionTitle = settings.tcf2!!.labelsPurposes,
            purposesOrSpecialFeatures = purposesProps,
            stacks = stacksProps,
        )
    }

    private fun specialPurposesCardsSection(): PredefinedUICardUISection? {
        if (tcfData.specialPurposes.isEmpty()) {
            return null
        }

        val purposesProps = UsercentricsMaps.mapSpecialPurposes(tcfData).map { TCFHolder(it) }
        val stacksProps = stacks.filter { it.stack.purposeIds.isNotEmpty() }.map { mapStackPropsToTCFHolder(it, it.stack.purposeIds, purposesProps) }

        return mapCardsSectionFromTCFHolder(
            sectionTitle = settings.tcf2!!.vendorSpecialPurposes,
            purposesOrSpecialFeatures = purposesProps,
            stacks = stacksProps,
        )
    }

    private fun specialFeaturesCardsSection(): PredefinedUICardUISection? {
        if (tcfData.specialFeatures.isEmpty()) {
            return null
        }

        val specialFeaturesProps = UsercentricsMaps.mapSpecialFeatures(tcfData).map { TCFHolder(it, hasToggles) }
        val stacksProps = stacks.filter { it.stack.specialFeatureIds.isNotEmpty() }.map { mapStackPropsToTCFHolder(it, it.stack.specialFeatureIds, specialFeaturesProps) }

        return mapCardsSectionFromTCFHolder(
            sectionTitle = settings.tcf2!!.vendorSpecialFeatures,
            purposesOrSpecialFeatures = specialFeaturesProps,
            stacks = stacksProps,
        )
    }

    private fun nonIABPurposesCardsSection(): PredefinedUICardUISection? {
        val skipNonIAB = settings.tcf2!!.hideNonIabOnFirstLayer
        if (skipNonIAB) {
            return null
        }

        if (categories.isEmpty()) {
            return null
        }

        val categoriesEntries = UsercentricsMaps.mapCategories(categories, services)

        if (categoriesEntries.isEmpty()){
            return null
        }

        return PredefinedUICardUISection(
            title = settings.tcf2.labelsNonIabPurposes,
            cards = categoriesEntries.map { entry ->
                if (!hasToggles) {
                    PredefinedUICardUI(
                        entry = entry, content = null, shortDescription = null, mainSwitchSettings = null
                    )
                } else {
                    PredefinedUICardUI(
                        entry = entry,
                        content = null,
                        shortDescription = null,
                    )
                }
            },
        )
    }

    private fun mapCardsSectionFromTCFHolder(sectionTitle: String, purposesOrSpecialFeatures: List<TCFHolder>, stacks: List<TCFHolder>): PredefinedUICardUISection {
        val purposesOrSpecialFeaturesAndStacks = purposesOrSpecialFeatures + stacks
        val purposesOrSpecialFeaturesCardsAndStacks = mutableListOf<PredefinedUICardUI>()

        purposesOrSpecialFeaturesAndStacks.forEach { tcfHolder ->
            val specialFeatureIsPartOfASelectedStack = tcfHolder.isPartOfASelectedStack
            if (specialFeatureIsPartOfASelectedStack) {
                return@forEach
            }

            val showContent = settings.tcf2!!.firstLayerShowDescriptions
            val cardContent = if (showContent) {
                PredefinedUISimpleCardContent(value = tcfHolder.contentDescription)
            } else {
                null
            }

            purposesOrSpecialFeaturesCardsAndStacks.add(
                PredefinedUICardUI(
                    tcfHolder,
                    cardContent = cardContent,
                )
            )
        }

        return PredefinedUICardUISection(
            title = sectionTitle,
            cards = purposesOrSpecialFeaturesCardsAndStacks,
        )
    }

    private fun mapStackPropsToTCFHolder(stackProps: StackProps, ids: List<Int>, propsHolderList: List<TCFHolder>): TCFHolder {
        return TCFHolder(
            stackProps = stackProps,
            hasMainToggle = hasToggles,
            dependantSwitchSettings = dependantSwitchSettingsOf(ids, propsHolderList)
        )
    }

    private fun dependantSwitchSettingsOf(purposeIds: List<Int>, propsHolderList: List<TCFHolder>): List<PredefinedUIDependantSwitchSettings> {
        return propsHolderList.filter { it.isPartOfASelectedStack && purposeIds.contains(it.tcfId) }
            .map { PredefinedUIDependantSwitchSettings(id = it.id, switchSettings = PredefinedUISwitchSettingsUI(it)) }
    }
}
