package com.usercentrics.sdk.v2.banner.service.mapper.tcf

import com.usercentrics.sdk.AdTechProvider
import com.usercentrics.sdk.UsercentricsAnalyticsEventType
import com.usercentrics.sdk.UsercentricsMaps
import com.usercentrics.sdk.extensions.isMultiple
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.models.tcf.TCFLabels
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.v2.banner.service.mapper.FooterButtonLayoutMapper
import com.usercentrics.sdk.v2.banner.service.mapper.GenericSecondLayerMapper
import com.usercentrics.sdk.v2.banner.service.mapper.PoweredByMapper
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.TCFStorageInformationHolder
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.TCFStorageInformationMapper
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.UsercentricsCategory
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization
import com.usercentrics.tcf.core.model.ConsentLanguages

internal class TCFSecondLayerMapper(
    private val settings: UsercentricsSettings,
    private val tcfData: TCFData,
    private val translations: LegalBasisLocalization,
    private val customization: PredefinedUICustomization,
    private val categories: List<UsercentricsCategory>,
    private val services: List<LegacyService>,
    private val labels: TCFLabels,
    private val controllerId: String,
    private val adTechProviders: List<AdTechProvider>,
) : GenericSecondLayerMapper(settings) {

    private val hasToggles: Boolean = !settings.tcf2!!.secondLayerHideToggles
    private val hideLegitimateInterestToggles: Boolean = settings.tcf2!!.hideLegitimateInterestToggles

    fun map() = UCUISecondLayerSettings(
        headerSettings = headerSettings(),
        footerSettings = footerSettings(),
        contentSettings = contentSettings()
    )

    fun mapTV(): PredefinedTVSecondLayerSettings {
        return PredefinedTVSecondLayerSettings(
            title = settings.tcf2!!.secondLayerTitle,
            save = settings.tcf2.buttonsSaveLabel,
            toggleLocalizations = PredefinedUIToggleLocalizations(
                onText = settings.tcf2.togglesSpecialFeaturesToggleOn,
                offText = settings.tcf2.togglesSpecialFeaturesToggleOff
            ),
            content = PredefinedTVSecondLayerSettingsContent.Cards(
                TCFSectionMapperTV(
                    tcfData = tcfData,
                    settings = settings,
                    translations = translations,
                    labels = labels,
                    categories = UsercentricsMaps.mapCategories(categories, services),
                    adTechProviders = adTechProviders,
                ).map()
            ),
            denyAll = denyAllText(),
            // TODO ADD ACCEPT ALL?????
        )
    }

    private fun denyAllText(): String? {
        val tcfSettings = settings.tcf2

        val shouldHideButtonDenyOnTcfSecondLayer = tcfSettings?.secondLayerHideButtonDeny == true

        return if (shouldHideButtonDenyOnTcfSecondLayer) {
            null
        } else tcfSettings?.buttonsDenyAllLabel
    }

    private fun headerSettings(): PredefinedUIHeaderSettings {
        return PredefinedUISecondLayerHeaderSettings(
            title = settings.tcf2!!.secondLayerTitle,
            links = headerLinks(),
            contentDescription = settings.tcf2.secondLayerDescription?.trim() ?: "",
            logoPosition = FirstLayerLogoPosition.LEFT,
            language = headerLanguageSettings(),
            logoURL = settings.customization?.logoUrl,
            firstLayerCloseLink = null,
            firstLayerCloseIcon = null,
        )
    }

    private fun headerLanguageSettings(): PredefinedUILanguageSettings? {
        val availableLanguages = mapAvailableLanguagesWithGVL(settings.languagesAvailable)

        val hideLanguage = !availableLanguages.isMultiple()

        return if (hideLanguage) {
            null
        } else {
            PredefinedUILanguageSettings(availableLanguages, PredefinedUILanguage(settings.language))
        }
    }

    private fun mapAvailableLanguagesWithGVL(languagesAvailable: List<String>): List<PredefinedUILanguage> {
        val matchedLanguages = mutableListOf<PredefinedUILanguage>()

        languagesAvailable.forEach { language ->
            if (ConsentLanguages.isLanguageAvailableOrSimilarDialectSupported(language)) {
                matchedLanguages.add(PredefinedUILanguage(language))
            }
        }
        return matchedLanguages.sortedBy { it.fullName }
    }

    private fun headerLinks(): List<PredefinedUILink> {
        val privacyLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.privacyPolicyLinkText,
            url = settings.privacyPolicyUrl,
            eventType = UsercentricsAnalyticsEventType.PRIVACY_POLICY_LINK,
        )
        val imprintLink = PredefinedUILink.legalLinkUrl(
            label = settings.labels.imprintLinkText,
            url = settings.imprintUrl,
            eventType = UsercentricsAnalyticsEventType.IMPRINT_LINK,
        )
        return listOf(privacyLink, imprintLink).filter { !it.isEmpty() }
    }

    private fun footerSettings(): PredefinedUIFooterSettings {
        val denyAllButton = if (settings.tcf2!!.secondLayerHideButtonDeny) {
            null
        } else {
            PredefinedUIFooterButton(
                label = settings.tcf2.buttonsDenyAllLabel,
                customization = customization.color.denyAllButton,
                type = PredefinedUIButtonType.DENY_ALL
            )
        }
        val acceptAllButton = PredefinedUIFooterButton(
            label = settings.tcf2.buttonsAcceptAllLabel,
            customization = customization.color.acceptAllButton,
            type = PredefinedUIButtonType.ACCEPT_ALL
        )
        val saveSettingsButton = if (hasToggles) PredefinedUIFooterButton(
            label = settings.tcf2.buttonsSaveLabel,
            customization = customization.color.saveButton,
            type = PredefinedUIButtonType.SAVE_SETTINGS
        ) else {
            null
        }
        val buttonLayoutMapper = FooterButtonLayoutMapper(
            acceptAll = acceptAllButton,
            denyAll = denyAllButton,
            saveSettings = saveSettingsButton,
        )
        return PredefinedUIFooterSettings(
            poweredBy = PoweredByMapper.mapPoweredBy(LegacyPoweredBy(isEnabled = settings.enablePoweredBy)),
            buttons = buttonLayoutMapper.mapButtons(),
            buttonsLandscape = buttonLayoutMapper.mapButtonsLandscape(),
        )
    }

    private fun contentSettings(): List<PredefinedUITabSettings> = listOf(
        purposesTab(),
        vendorsTab()
    )

    private fun purposesTab(): PredefinedUITabSettings {
        val cardsSection = mutableListOf<PredefinedUICardUISection>()

        val purposesCardsSections = purposesCardsSection()
        if (purposesCardsSections != null) {
            cardsSection.add(purposesCardsSections)
        }

        val specialPurposesCardsSections = specialPurposesCardsSection()
        if (specialPurposesCardsSections != null) {
            cardsSection.add(specialPurposesCardsSections)
        }

        val featuresCardsSections = featuresCardsSection()
        if (featuresCardsSections != null) {
            cardsSection.add(featuresCardsSections)
        }

        val specialFeatureCardsSection = specialFeaturesCardsSection()
        if (specialFeatureCardsSection != null) {
            cardsSection.add(specialFeatureCardsSection)
        }

        val nonIABCardsSections = nonIABCardsSection()
        if (nonIABCardsSections != null) {
            cardsSection.add(nonIABCardsSections)
        }

        return PredefinedUITabSettings(
            title = settings.tcf2!!.tabsPurposeLabel,
            content = PredefinedUICategoriesContent(cardUISections = cardsSection)
        )
    }

    private fun purposesCardsSection(): PredefinedUICardUISection? {
        val purposesCards = purposesCards()
        if (purposesCards.isEmpty()) {
            return null
        }
        return PredefinedUICardUISection(
            title = settings.tcf2!!.labelsPurposes,
            cards = purposesCards
        )
    }

    private fun specialPurposesCardsSection(): PredefinedUICardUISection? {
        val specialPurposesCards = specialPurposesCards()
        if (specialPurposesCards.isEmpty()) {
            return null
        }
        return PredefinedUICardUISection(
            title = settings.tcf2!!.vendorSpecialPurposes,
            cards = specialPurposesCards
        )
    }

    private fun purposesCards(): List<PredefinedUICardUI> {
        if (tcfData.purposes.isEmpty()) {
            return listOf()
        }
        val purposesProps = UsercentricsMaps.mapPurposes(tcfData).map { TCFHolder(it, hasMainToggle = false, hideLegitimateInterestToggles = hideLegitimateInterestToggles) }
        return purposesProps.map { tcfHolder ->
            PredefinedUICardUI(
                tcfHolder = tcfHolder,
                cardContent = PredefinedUIPurposeCardContent(
                    examples = PredefinedUISimpleCardContent(
                        title = settings.tcf2!!.examplesLabel,
                        description = tcfHolder.contentDescription,
                        values = tcfHolder.illustrations,
                    ),
                    vendors = PredefinedUIPurposeVendorDetails(
                        title = settings.tcf2.tabsVendorsLabel,
                        value = tcfHolder.numberOfVendors.toString(),
                    )
                ),
                switchSettings = if (hasToggles) {
                    contentSwitchSettingsRow(tcfHolder)
                } else {
                    null
                },
            )
        }
    }

    private fun specialPurposesCards(): List<PredefinedUICardUI> {
        val specialPurposes = tcfData.specialPurposes
        if (specialPurposes.isEmpty()) {
            return listOf()
        }
        return specialPurposes.map { specialPurpose -> PredefinedUICardUI(specialPurpose, settings.tcf2!!.examplesLabel) }
    }

    private fun featuresCardsSection(): PredefinedUICardUISection? {
        val featuresCards = featuresCards()
        if (featuresCards.isEmpty()) {
            return null
        }
        return PredefinedUICardUISection(
            title = settings.tcf2!!.labelsFeatures,
            cards = featuresCards
        )
    }

    private fun specialFeaturesCardsSection(): PredefinedUICardUISection? {
        val specialFeaturesCards = specialFeaturesCards()
        if (specialFeaturesCards.isEmpty()) {
            return null
        }
        return PredefinedUICardUISection(
            title = settings.tcf2!!.vendorSpecialFeatures,
            cards = specialFeaturesCards
        )
    }

    private fun featuresCards(): List<PredefinedUICardUI> {
        val features = tcfData.features
        if (features.isEmpty()) {
            return listOf()
        }
        return features.map { feature -> PredefinedUICardUI(feature, settings.tcf2!!.examplesLabel) }
    }

    private fun specialFeaturesCards(): List<PredefinedUICardUI> {
        val specialFeatures = tcfData.specialFeatures
        if (specialFeatures.isEmpty()) {
            return listOf()
        }

        val specialFeaturesHolder = UsercentricsMaps.mapSpecialFeatures(tcfData).map { TCFHolder(it, hasMainToggle = hasToggles) }
        return specialFeaturesHolder.map { tcfHolder ->
            PredefinedUICardUI(
                tcfHolder = tcfHolder,
                cardContent = PredefinedUISimpleCardContent(
                    title = settings.tcf2!!.examplesLabel,
                    description = tcfHolder.contentDescription,
                    values = tcfHolder.illustrations
                ),
                switchSettings = null
            )
        }
    }

    private fun nonIABCardsSection(): PredefinedUICardUISection? {
        val nonIABCards = nonIABCards()
        if (nonIABCards.isEmpty()) {
            return null
        }
        return PredefinedUICardUISection(
            title = settings.tcf2!!.labelsNonIabPurposes,
            cards = nonIABCards
        )
    }

    private fun nonIABCards(): List<PredefinedUICardUI> {
        if (categories.isEmpty()) {
            return listOf()
        }
        val categoriesEntries = UsercentricsMaps.mapCategories(categories, services)

        return categoriesEntries
            .map { entry ->
                if (!hasToggles) {
                    PredefinedUICardUI(
                        entry = entry,
                        content = PredefinedUIServicesCardContent(
                            services = entry.services.map { service ->
                                PredefinedUIServiceDetails(
                                    service,
                                    hideTogglesForServices = true,
                                    dpsDisplayFormat = settings.dpsDisplayFormat,
                                    consent = predefinedUIServiceConsent(service.consent),
                                )
                            }
                        ),
                        shortDescription = entry.category.description,
                        mainSwitchSettings = null,
                    )
                } else {
                    PredefinedUICardUI(
                        entry = entry,
                        content = PredefinedUIServicesCardContent(
                            services = entry.services.map { service ->
                                PredefinedUIServiceDetails(
                                    service = service,
                                    dpsDisplayFormat = settings.dpsDisplayFormat,
                                    consent = predefinedUIServiceConsent(service.consent),
                                )
                            }
                        ),
                        shortDescription = entry.category.description
                    )
                }
            }
    }

    private fun vendorsTab(): PredefinedUITabSettings {
        val cardsSection = mutableListOf<PredefinedUICardUISection>()

        val vendorsCardsSections = vendorsCardsSection()
        if (vendorsCardsSections != null) {
            cardsSection.add(vendorsCardsSections)
        }

        val nonIABVendorsCardsSections = nonIABVendorsCardsSection()
        if (nonIABVendorsCardsSections != null) {
            cardsSection.add(nonIABVendorsCardsSections)
        }

        val adTechProvidersCardsSection = adTechProvidersCardsSection()
        if (adTechProvidersCardsSection != null) {
            cardsSection.add(adTechProvidersCardsSection)
        }

        cardsSection.lastOrNull()?.let { lastSection: PredefinedUICardUISection ->
            val cards = lastSection.cards.toMutableList()
            cards.add(cmpMaxDurationStorage())

            val updatedLastSection = lastSection.copy(cards = cards, controllerID = controllerIDSettings())
            cardsSection[cardsSection.lastIndex] = updatedLastSection
        }

        return PredefinedUITabSettings(
            title = settings.tcf2!!.tabsVendorsLabel,
            content = PredefinedUIServicesContent(cardUISections = cardsSection)
        )
    }

    private fun cmpMaxDurationStorage(): PredefinedUICardUI {
        return PredefinedUICardUI(
            id = "",
            title = translations.labels.tcfMaxDurationTitle,
            shortDescription = null,
            mainSwitchSettings = null,
            content = PredefinedUISimpleCardContent(
                title = "",
                description = translations.labels.tcfMaxDurationText,
                value = ""
            ),
            switchSettings = null,
            dependantSwitchSettings = null
        )
    }

    private fun adTechProvidersCardsSection(): PredefinedUICardUISection? {
        if (adTechProviders.isEmpty()) {
            return null
        }

        val cards = adTechProviders.map {
            PredefinedUICardUI(
                id = ServicesIdStrategy.id(it),
                title = it.name,
                shortDescription = "",
                content = PredefinedUISingleServiceCardContent(
                    service = PredefinedUIServiceDetails(adTechProvider = it),
                ),
                mainSwitchSettings = PredefinedUISwitchSettingsUI(
                    id = PredefinedUIDecision.CONSENT_ID,
                    disabled = false,
                    currentValue = it.consent
                ),
            )
        }
        return PredefinedUICardUISection(
            title = settings.tcf2?.atpListTitle?.let { formatMixedDirectionString(it, cards.size.toString()) },
            cards = cards
        )
    }

    private fun vendorsCardsSection(): PredefinedUICardUISection? {
        if (tcfData.vendors.isEmpty()) {
            return null
        }

        val vendorMapperList = UsercentricsMaps.mapVendors(tcfData).map { TCFVendorMapper(it, settings, labels) }
        val cards = vendorMapperList.map { vendorMapper ->
            val tcfHolder = vendorMapper.tcfHolder
            PredefinedUICardUI(
                tcfHolder,
                cardContent = PredefinedUISingleServiceCardContent(service = vendorMapper.mapServiceDetails()),
                switchSettings = if (hasToggles) contentSwitchSettingsRow(tcfHolder) else null
            )
        }
        return PredefinedUICardUISection(
            title = formatMixedDirectionString(settings.tcf2!!.labelsIabVendors, cards.size.toString()),
            cards = cards
        )
    }

    /**
     * Prepares a localized string that supports both RTL and LTR languages, ensuring proper text direction for mixed content.
     * It embeds RTL text (e.g., Arabic, Hebrew) with LTR segments (e.g., numbers, English).
     * @return A correctly formatted string with proper directional control for RTL and LTR text.
     */
    private fun formatMixedDirectionString(label: String, cardsSize: String): String {
        return "\u202B${label}\u202C \u202A(${cardsSize})\u202C"
    }

    private fun contentSwitchSettingsRow(tcfHolder: TCFHolder): List<PredefinedUISwitchSettingsUI> {
        val contentSwitchSettings = mutableListOf<PredefinedUISwitchSettingsUI>()
        if (tcfHolder.showConsentToggle) {
            contentSwitchSettings.add(
                PredefinedUISwitchSettingsUI(
                    id = PredefinedUIDecision.CONSENT_ID,
                    label = settings.tcf2!!.togglesConsentToggleLabel,
                    disabled = false,
                    currentValue = tcfHolder.consentValue
                )
            )
        }
        if (tcfHolder.showLegitimateInterestToggle) {
            contentSwitchSettings.add(
                PredefinedUISwitchSettingsUI(
                    id = PredefinedUIDecision.LEGITIMATE_INTEREST_ID,
                    label = settings.tcf2!!.togglesLegIntToggleLabel,
                    disabled = false,
                    currentValue = tcfHolder.legitimateInterestValue
                )
            )
        }
        return contentSwitchSettings
    }

    private fun nonIABVendorsCardsSection(): PredefinedUICardUISection? {
        if (services.isEmpty()) {
            return null
        }

        val servicesToShow = services.filter { !it.isHidden }
        val cards = servicesToShow.map { service ->

            PredefinedUICardUI(
                service = service,
                content = PredefinedUISingleServiceCardContent(
                    service = PredefinedUIServiceDetails(
                        service = service,
                        storageInformationServiceContent = storageInformationSection(service),
                        dpsDisplayFormat = settings.dpsDisplayFormat,
                        consent = predefinedUIServiceConsent(service.consent),
                    ),
                ),
                mainSwitchSettings = if (hasToggles) PredefinedUISwitchSettingsUI(
                    id = PredefinedUIDecision.CONSENT_ID,
                    disabled = service.isEssential,
                    currentValue = service.consent.status
                ) else null
            )
        }
        return PredefinedUICardUISection(
            title = formatMixedDirectionString(settings.tcf2!!.labelsNonIabVendors, cards.size.toString()),
            cards = cards
        )
    }

    private fun storageInformationSection(service: LegacyService): PredefinedUIServiceContentSection? {
        if (service.usesNonCookieAccess == null && service.cookieMaxAgeSeconds == null) {
            return null
        }

        val tcfStorageInformationMapper = TCFStorageInformationMapper(
            holder = TCFStorageInformationHolder(
                cookieMaxAgeSeconds = service.cookieMaxAgeSeconds,
                usesNonCookieAccess = service.usesNonCookieAccess,
                deviceStorageDisclosureUrl = service.deviceStorageDisclosureUrl,
                deviceStorage = service.deviceStorage,
                usesCookies = null,
                cookieRefresh = null,
                cookieInformationLabels = labels.cookieInformation,
            ),
            showShortDescription = true
        )
        return tcfStorageInformationMapper.map()
    }

    private fun controllerIDSettings(): PredefinedUIControllerIDSettings {
        return PredefinedUIControllerIDSettings(
            label = translations.labels.controllerIdTitle,
            value = controllerId
        )
    }
}
