package com.usercentrics.sdk.v2.banner.service.mapper.tcf

import com.usercentrics.sdk.*
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.models.tcf.TCFLabels
import com.usercentrics.sdk.services.tcf.interfaces.TCFData
import com.usercentrics.sdk.v2.banner.service.mapper.gdpr.GDPRSectionMapperTV
import com.usercentrics.sdk.v2.settings.data.TCF2Settings
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.translation.data.LegalBasisLocalization

internal class TCFSectionMapperTV(
    private val tcfData: TCFData,
    private val settings: UsercentricsSettings,
    private val translations: LegalBasisLocalization,
    private val labels: TCFLabels,
    private val categories: List<CategoryProps>,
    private val adTechProviders: List<AdTechProvider>,
) {

    private val detailsLabel: String
        get() = translations.labels.details

    private val tcf2: TCF2Settings
        get() = settings.tcf2!!

    fun map(): List<PredefinedTVSecondLayerCardSection> {
        return listOfNotNull(
            tvPurposesSection(tcfData),
            tvFeaturesSection(),
            tvSpecialFeaturesSection(),
            tvNonIABSection(),
            tvVendorsSection(),
            tvAdTechProvidersSection(),
        )
    }

    private fun tvPurposesSection(tcfData: TCFData): PredefinedTVSecondLayerCardSection? {
        val purposes = UsercentricsMaps.mapPurposes(tcfData).map {
            val consentToggle = if (it.purpose.showConsentToggle) PredefinedUIToggleSettings(
                id = ServicesIdStrategy.id(it.purpose),
                title = settings.labels.consent,
                consentId = PredefinedUIDecision.CONSENT_ID,
                isEnabled = true,
                currentValue = it.checked,
                dependentsIds = emptyList(),
                parentId = null
            ) else null
            PredefinedTVSecondLayerCardEntry(
                id = "PurposesSection-${it.purpose.id}",
                title = it.purpose.name,
                description = null,
                toggle = consentToggle,
                content = tvPurposeContent(it)
            )
        }

        val specialPurposes = tcfData.specialPurposes.map {
            PredefinedTVSecondLayerCardEntry(
                id = "SpecialPurposesSection-${it.id}",
                title = it.name,
                description = null,
                toggle = null,
                content = PredefinedTVSecondLayerSettingsContent.DetailsWithIllustrations(
                    extraToggle = null,
                    entries = mapEntriesWithIllustrations(it.purposeDescription, it.illustrations)
                )
            )
        }

        val entries = purposes + specialPurposes
        if (entries.isEmpty()) {
            return null
        }

        return PredefinedTVSecondLayerCardSection(
            name = tcf2.labelsPurposes,
            entries = entries
        )
    }

    private fun tvPurposeContent(property: PurposeProps): PredefinedTVSecondLayerSettingsContent {
        val legitimateInterestToggle = if (property.purpose.showLegitimateInterestToggle) {
            PredefinedUIToggleSettings(
                id = ServicesIdStrategy.id(property.purpose),
                title = tcf2.togglesLegIntToggleLabel,
                consentId = PredefinedUIDecision.LEGITIMATE_INTEREST_ID,
                isEnabled = true,
                currentValue = property.legitimateInterestChecked,
                dependentsIds = emptyList(),
                parentId = null
            )
        } else {
            null
        }

        return PredefinedTVSecondLayerSettingsContent.DetailsWithIllustrations(
            extraToggle = legitimateInterestToggle,
            entries = mapEntriesWithIllustrations(property.purpose.purposeDescription, property.purpose.illustrations, property.purpose.numberOfVendors),
        )
    }

    private fun tvFeaturesSection(): PredefinedTVSecondLayerCardSection? {
        val features = tcfData.features.map {
            PredefinedTVSecondLayerCardEntry(
                id = "FeaturesSection-${it.id}",
                title = it.name,
                description = null,
                toggle = null,
                content = PredefinedTVSecondLayerSettingsContent.DetailsWithIllustrations(
                    extraToggle = null,
                    entries = mapEntriesWithIllustrations(it.purposeDescription, it.illustrations)
                )
            )
        }

        val entries = features
        if (entries.isEmpty()) return null

        return PredefinedTVSecondLayerCardSection(
            name = tcf2.labelsFeatures,
            entries = entries
        )
    }

    private fun tvSpecialFeaturesSection(): PredefinedTVSecondLayerCardSection? {

        val specialFeatures = UsercentricsMaps.mapSpecialFeatures(tcfData).map {
            val consentToggle = PredefinedUIToggleSettings(
                id = ServicesIdStrategy.id(it.specialFeature),
                title = settings.labels.consent,
                consentId = PredefinedUIDecision.CONSENT_ID,
                isEnabled = true,
                currentValue = it.checked,
                dependentsIds = emptyList(),
                parentId = null
            )
            PredefinedTVSecondLayerCardEntry(
                id = "SpecialFeaturesSection-${it.specialFeature.id}",
                title = it.specialFeature.name,
                description = null,
                toggle = consentToggle,
                content = PredefinedTVSecondLayerSettingsContent.DetailsWithIllustrations(
                    extraToggle = null,
                    entries = mapEntriesWithIllustrations(it.specialFeature.purposeDescription, it.specialFeature.illustrations)
                )
            )
        }

        val entries = specialFeatures
        if (entries.isEmpty()) return null

        return PredefinedTVSecondLayerCardSection(
            name = tcf2.vendorSpecialFeatures,
            entries = entries
        )
    }

    private fun mapEntriesWithIllustrations(description: String, illustrations: List<String>, numberOfVendors: Int? = null): List<PredefinedTVSecondLayerDetailsEntry> {
        val entries = mutableListOf<PredefinedTVSecondLayerDetailsEntry>(
            PredefinedTVSecondLayerDetailsEntry.TitleContent(title = detailsLabel, content = description)
        )

        if (illustrations.isNotEmpty()) {
            entries.add(
                PredefinedTVSecondLayerDetailsEntry.TitleContent(
                    title = settings.tcf2!!.examplesLabel,
                    content = PredefinedTVSecondLayerDetailsEntry.mapContentIllustrations(illustrations),
                )
            )
        }

        if (numberOfVendors != null) {
            entries.add(
                PredefinedTVSecondLayerDetailsEntry.Text(
                    text = settings.tcf2!!.tabsVendorsLabel + ": " + numberOfVendors
                )
            )
        }

        return entries
    }

    private fun tvNonIABSection(): PredefinedTVSecondLayerCardSection {
        return GDPRSectionMapperTV(
            categoriesSectionLabel = tcf2.labelsNonIabPurposes,
            servicesSectionLabel = tcf2.labelsNonIabVendors,
            consentLabel = settings.labels.consent,
            hideTogglesForServices = false,
            settings = settings,
            serviceLabels = labels.nonTCFLabels.service,
            translations = translations
        ).map(categories)
    }

    private fun tvVendorsSection(): PredefinedTVSecondLayerCardSection? {
        if (tcfData.vendors.isEmpty()) return null

        return PredefinedTVSecondLayerCardSection(
            name = tcf2.labelsIabVendors,
            entries = UsercentricsMaps.mapVendors(tcfData).map {
                val consentToggle = if (it.vendor.showConsentToggle) {
                    PredefinedUIToggleSettings(
                        id = ServicesIdStrategy.id(it.vendor),
                        title = settings.labels.consent,
                        consentId = PredefinedUIDecision.CONSENT_ID,
                        isEnabled = true,
                        currentValue = it.checked,
                        dependentsIds = emptyList(),
                        parentId = null
                    )
                } else null

                PredefinedTVSecondLayerCardEntry(
                    id = "VendorsSection-${it.vendor.id}",
                    title = it.vendor.name,
                    description = null,
                    toggle = consentToggle,
                    content = tvVendorContent(it)
                )
            }
        )
    }

    private fun tvVendorContent(property: VendorProps): PredefinedTVSecondLayerSettingsContent {
        val legitimateToggle = if (property.vendor.showLegitimateInterestToggle) {
            PredefinedUIToggleSettings(
                id = ServicesIdStrategy.id(property.vendor),
                title = tcf2.togglesLegIntToggleLabel,
                consentId = PredefinedUIDecision.LEGITIMATE_INTEREST_ID,
                isEnabled = true,
                currentValue = property.legitimateInterestChecked,
                dependentsIds = emptyList(),
                parentId = null
            )
        } else {
            null
        }

        return PredefinedTVSecondLayerSettingsContent.DetailsWithVendors(
            title = detailsLabel,
            extraToggle = legitimateToggle,
            entries = TCFDetailsMapperTV(settings = settings, labels = labels).map(property.vendor)
        )
    }

    private fun tvAdTechProvidersSection(): PredefinedTVSecondLayerCardSection? {
        if (adTechProviders.isEmpty()) {
            return null
        }

        val entries = adTechProviders.map {
            PredefinedTVSecondLayerCardEntry(
                id = ServicesIdStrategy.id(it),
                title = it.name,
                description = "",

                toggle = PredefinedUIToggleSettings(
                    id = ServicesIdStrategy.id(it),
                    title = it.name,
                    consentId = PredefinedUIDecision.CONSENT_ID,
                    isEnabled = true,
                    currentValue = it.consent,
                    dependentsIds = emptyList(),
                    parentId = null
                ),

                content = PredefinedTVSecondLayerSettingsContent.DetailsWithVendors(
                    title = "",
                    extraToggle = null,
                    entries = listOf(
                        PredefinedTVSecondLayerDetailsEntry.Link(
                            title = settings.labels.privacyPolicyLinkText,
                            linkLabel = it.privacyPolicyUrl,
                            url = it.privacyPolicyUrl,
                        )
                    )
                )
            )
        }

        return PredefinedTVSecondLayerCardSection(
            name = "Google Ad Technology Providers (ATPs)",
            entries = entries
        )
    }
}
