package com.usercentrics.sdk.v2.banner.service.mapper.tcf

import com.usercentrics.sdk.VendorProps
import com.usercentrics.sdk.models.settings.*
import com.usercentrics.sdk.models.tcf.TCFLabels
import com.usercentrics.sdk.services.tcf.interfaces.IdAndName
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.TCFStorageInformationHolder
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.TCFStorageInformationMapper
import com.usercentrics.sdk.v2.banner.service.mapper.tcf.storageinfo.getVendorUrls
import com.usercentrics.sdk.v2.settings.data.TCF2Settings
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.tcf.core.model.gvl.RetentionPeriod

internal class TCFVendorMapper(
    vendorProps: VendorProps,
    private val settings: UsercentricsSettings,
    private val labels: TCFLabels
) {

    private val tcf2Settings: TCF2Settings
        get() = settings.tcf2!!

    val tcfHolder = TCFHolder(vendorProps, hideLegitimateInterestToggles = settings.tcf2!!.hideLegitimateInterestToggles)
    private val vendor = vendorProps.vendor

    fun mapServiceDetails(): PredefinedUIServiceDetails {
        return PredefinedUIServiceDetails(
            id = tcfHolder.id,
            serviceContentSection = mapServiceContentSection()
        )
    }

    private fun mapServiceContentSection(): List<PredefinedUIServiceContentSection> {
        return listOfNotNull(
            purposesProcessedByConsent,
            legitimateInterestPurposes,
            specialPurposes,
            features,
            specialFeatures,
            categoriesOfData,
            dataSharedOutsideEU,
            policyURL,
            legitimateInterestURL,
            storageInformation,
            dataRetentionPeriod,
        )
    }

    private val dataSharedOutsideEU: PredefinedUIServiceContentSection?
        get() {
            val title = settings.labels.transferToThirdCountries
            val description = labels.vendorsOutsideEU
            val hasInfo = title.isNotBlank() && description.isNotBlank()
            return if (vendor.dataSharedOutsideEU == true && hasInfo) {
                PredefinedUIServiceContentSection(
                    title = title,
                    content = PredefinedUISimpleServiceContent(description),
                )
            } else {
                null
            }
        }

    private val purposesProcessedByConsent: PredefinedUIServiceContentSection? =
        bulletServiceContentSection(
            title = tcf2Settings.vendorPurpose,
            idAndNameList = vendor.purposes,
            retentionPeriod = vendor.dataRetention?.purposes,
        )

    private val categoriesOfData: PredefinedUIServiceContentSection? = bulletServiceContentSection(
        tcf2Settings.categoriesOfDataLabel,
        vendor.dataCategories
    )

    private val dataRetentionPeriod: PredefinedUIServiceContentSection? = setDataRetentionPeriod(
        tcf2Settings.dataRetentionPeriodLabel,
        vendor.dataRetention?.stdRetention
    )

    private val legitimateInterestPurposes: PredefinedUIServiceContentSection? =
        bulletServiceContentSection(
            tcf2Settings.vendorLegitimateInterestPurposes,
            vendor.legitimateInterestPurposes
        )


    private val specialPurposes: PredefinedUIServiceContentSection? =
        bulletServiceContentSection(
            title = tcf2Settings.vendorSpecialPurposes,
            idAndNameList = vendor.specialPurposes,
            retentionPeriod = vendor.dataRetention?.specialPurposes,
        )

    private val features: PredefinedUIServiceContentSection? =
        bulletServiceContentSection(tcf2Settings.vendorFeatures, vendor.features)

    private val specialFeatures: PredefinedUIServiceContentSection? =
        bulletServiceContentSection(tcf2Settings.vendorSpecialFeatures, vendor.specialFeatures)

    private val policyURL: PredefinedUIServiceContentSection?
        get() {
            val policyUrl = getVendorUrls(vendor, settings)?.privacy ?: return null

            return PredefinedUIServiceContentSection(
                settings.labels.privacyPolicyLinkText,
                content = PredefinedUIHyperlinkServiceContent(policyUrl)
            )
        }

    private val legitimateInterestURL: PredefinedUIServiceContentSection?
        get() {
            val legIntClaimUrl = getVendorUrls(vendor, settings)?.legIntClaim ?: return null

            return PredefinedUIServiceContentSection(
                tcf2Settings.legitimateInterestLabel,
                content = PredefinedUIHyperlinkServiceContent(legIntClaimUrl)
            )
        }

    private val storageInformation: PredefinedUIServiceContentSection
        get() {
            val tcfStorageInformationMapper = TCFStorageInformationMapper(
                holder = TCFStorageInformationHolder(
                    cookieMaxAgeSeconds = vendor.cookieMaxAgeSeconds?.toLong(),
                    usesNonCookieAccess = vendor.usesNonCookieAccess,
                    deviceStorageDisclosureUrl = vendor.deviceStorageDisclosureUrl,
                    usesCookies = vendor.usesCookies,
                    cookieRefresh = vendor.cookieRefresh,
                    cookieInformationLabels = labels.cookieInformation,
                ),
                showShortDescription = false
            )
            return tcfStorageInformationMapper.map()
        }

    private fun bulletServiceContentSection(title: String, idAndNameList: List<IdAndName>, retentionPeriod: RetentionPeriod? = null): PredefinedUIServiceContentSection? {
        val content = idAndNameList.joinToString(separator = "\n") { idAndName ->
            val time = retentionPeriod?.idAndPeriod?.get(idAndName.id)
            if (time != null) {
                "${PredefinedUIData.BULLET_POINT} ${idAndName.name.trim()} (${tcf2Settings.dataRetentionPeriodLabel}: ${time})"
            } else {
                "${PredefinedUIData.BULLET_POINT} ${idAndName.name.trim()}"
            }
        }

        if (content.isBlank()) {
            return null
        }
        return PredefinedUIServiceContentSection(
            title = title,
            content = PredefinedUISimpleServiceContent(content)
        )
    }

    private fun setDataRetentionPeriod(title: String, dataRetentionPeriod: Int?): PredefinedUIServiceContentSection? {
        if (dataRetentionPeriod == null) {
            return null
        }

        val content = "${PredefinedUIData.BULLET_POINT} $dataRetentionPeriod"

        return PredefinedUIServiceContentSection(
            title = title,
            content = PredefinedUISimpleServiceContent(content)
        )
    }
}


