package com.usercentrics.sdk.v2.consent.api

import com.usercentrics.sdk.BuildKonfig
import com.usercentrics.sdk.core.hash.UUID
import com.usercentrics.sdk.core.json.JsonParser
import com.usercentrics.sdk.domain.api.http.HttpRequests
import com.usercentrics.sdk.extensions.secondsToMillis
import com.usercentrics.sdk.services.api.NetworkResolver
import com.usercentrics.sdk.ui.userAgent.UserAgentProvider
import com.usercentrics.sdk.ui.userAgent.UsercentricsUserAgentInfo
import com.usercentrics.sdk.v2.consent.data.*

internal class SaveConsentsApiImpl(
    private val requests: HttpRequests,
    private val networkResolver: NetworkResolver,
    private val jsonParser: JsonParser,
    private val userAgentProvider: UserAgentProvider,
) : SaveConsentsApi {

    private val userAgentInfo by lazy { userAgentProvider.provide() }

    override fun saveConsents(
        consentsData: SaveConsentsData,
        analyticsFlag: Boolean,
        xdeviceFlag: Boolean,
        consentWebhook: Boolean,
        onSuccess: () -> Unit,
        onError: (Throwable) -> Unit,
    ) {
        requests.post(
            url = getSaveEndpointUrl(),
            bodyData = createPayload(
                consentsData = consentsData,
                xdeviceFlag = xdeviceFlag,
                analyticsFlag = analyticsFlag,
                consentWebhook = consentWebhook
            ),
            headers = getHeaders(),
            onSuccess = {
                onSuccess()
            },
            onError = onError
        )
    }

    // TODO Do we need this?
    private fun getHeaders(): Map<String, String> {
        val allowedOriginDomain = BuildKonfig.endpoint_saveConsentAllowedOriginDomain
        return mapOf(
            "Accept" to "application/json",
            "Access-Control-Allow-Origin" to allowedOriginDomain,
            "X-Request-ID" to UUID.random()
        )
    }

    private fun createPayload(
        consentsData: SaveConsentsData,
        analyticsFlag: Boolean,
        xdeviceFlag: Boolean,
        consentWebhook: Boolean
    ): String {
        val dto = consentsData.mapDto(
            userAgentInfo = userAgentInfo,
            analyticsFlag = analyticsFlag,
            xdeviceFlag = xdeviceFlag,
            consentWebhook = consentWebhook
        )
        return jsonParser.encodeToString(SaveConsentsDto.serializer(), dto)
    }

    private fun getSaveEndpointUrl(): String {
        return "${networkResolver.saveConsentsBaseUrl()}/consent/ua/3"
    }

    private fun SaveConsentsData.mapDto(
        userAgentInfo: UsercentricsUserAgentInfo,
        analyticsFlag: Boolean,
        xdeviceFlag: Boolean,
        consentWebhook: Boolean,
    ): SaveConsentsDto {
        val consentString = consentStringObject?.string ?: ""
        val consentMetaPayload = ConsentStringObjectDto.create(dataTransferObject.timestampInSeconds.secondsToMillis(), consentStringObject)
        val consentMeta = if (consentMetaPayload == null) "" else jsonParser.encodeToString(ConsentStringObjectDto.serializer(), consentMetaPayload)
        return SaveConsentsDto(
            action = dataTransferObject.consent.action.text,
            appVersion = userAgentInfo.appVersion,
            controllerId = dataTransferObject.settings.controllerId,
            language = dataTransferObject.settings.language,
            settingsId = dataTransferObject.settings.id,
            settingsVersion = dataTransferObject.settings.version,
            consentString = consentString,
            consentMeta = consentMeta,
            consents = dataTransferObject.services.map { it.mapDto() },
            bundleId = userAgentInfo.appID,
            sdkVersion = userAgentInfo.sdkVersion,
            userOS = userAgentInfo.platform,
            xdevice = xdeviceFlag,
            analytics = analyticsFlag,
            acString = this.acString ?: "",
            webhook = consentWebhook
        )
    }

    private fun DataTransferObjectService.mapDto(): ConsentStatusDto {
        return ConsentStatusDto(
            consentStatus = status,
            consentTemplateId = id,
            consentTemplateVersion = version,
        )
    }
}
