package com.usercentrics.sdk.v2.consent.service

import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.models.settings.UsercentricsConsentAction
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.services.deviceStorage.models.ConsentsBuffer
import com.usercentrics.sdk.services.deviceStorage.models.ConsentsBufferEntry
import com.usercentrics.sdk.services.settings.ISettingsLegacy
import com.usercentrics.sdk.v2.async.dispatcher.Dispatcher
import com.usercentrics.sdk.v2.consent.api.GetConsentsApi
import com.usercentrics.sdk.v2.consent.api.SaveConsentsApi
import com.usercentrics.sdk.v2.consent.data.ConsentStringObject
import com.usercentrics.sdk.v2.consent.data.DataTransferObject
import com.usercentrics.sdk.v2.consent.data.GetConsentsData
import com.usercentrics.sdk.v2.consent.data.SaveConsentsData
import com.usercentrics.sdk.v2.settings.data.UsercentricsSettings
import com.usercentrics.sdk.v2.settings.service.ISettingsService

internal class ConsentsServiceImpl(
    private val dispatcher: Dispatcher,
    private val logger: UsercentricsLogger,
    private val getConsentsApi: GetConsentsApi,
    private val saveConsentsApi: SaveConsentsApi,
    private val deviceStorage: DeviceStorage,
    private val settingsService: ISettingsService,
    private val settingsLegacyInstance: ISettingsLegacy,
) : ConsentsService {

    private val settings: UsercentricsSettings
        get() = settingsService.settings?.data ?: throw IllegalStateException("Consents Service requires a valid Settings state")

    private val analyticsFlag: Boolean
        get() = settings.consentAnalytics

    private val xdeviceFlag: Boolean
        get() = settings.consentXDevice

    private val consentWebhook: Boolean
        get() = settings.consentWebhook

    override fun getRemoteUserConsents(controllerId: String, onSuccess: (GetConsentsData) -> Unit, onError: (UsercentricsException) -> Unit) {
        getConsentsApi.getUserConsents(controllerId, onSuccess, onError)
    }

    override fun saveConsentsState(cause: UsercentricsConsentAction) {
        // FIXME: stream consents only after a successful initialization
        dispatcher.dispatch {
            val consentsData = createState(cause)
            doSaveConsents(consentsData)
        }
    }

    private fun doSaveConsents(consentsData: SaveConsentsData) {
        saveConsentsApi.saveConsents(
            consentsData = consentsData,
            analyticsFlag = analyticsFlag,
            xdeviceFlag = xdeviceFlag,
            consentWebhook = consentWebhook,
            onSuccess = {
                clearConsentsFromBuffer(consentsData)
            },
            onError = {
                logger.error("Failed while trying to save consents", it)
                addConsentsToBuffer(consentsData)
            }
        )
    }

    private fun createState(cause: UsercentricsConsentAction): SaveConsentsData {
        // This is a workaround for:
        // - [MSDK-1739] Consents V2 API Optimizations
        // - [MSDK-1708] Save non-IAB consents and the TCString in the same request
        return if (cause == UsercentricsConsentAction.TCF_STRING_CHANGE) {
            createStateForTCF(cause)
        } else {
            createStateForGDPR(cause)
        }
    }

    private fun createStateForGDPR(cause: UsercentricsConsentAction): SaveConsentsData {
        val dataTransferObject = DataTransferObject.create(
            settings = settings,
            controllerId = settingsLegacyInstance.getSettings().controllerId,
            services = settingsLegacyInstance.getSettings().services,
            consentAction = cause,
            consentType = cause.getType()
        )

        return SaveConsentsData(
            dataTransferObject = dataTransferObject,
            consentStringObject = null
        )
    }

    private fun createStateForTCF(cause: UsercentricsConsentAction): SaveConsentsData {
        val dataTransferObject = DataTransferObject.create(
            settings = settings,
            controllerId = settingsLegacyInstance.getSettings().controllerId,
            services = emptyList(),
            consentAction = cause,
            consentType = cause.getType()
        )

        // FIXME: consume the acString directly from TCF Storage, called in createConsentStringState()
        val acString = deviceStorage.getACString()

        return SaveConsentsData(
            dataTransferObject = dataTransferObject,
            consentStringObject = createConsentStringState(),
            acString = acString,
        )
    }

    private fun createConsentStringState(): ConsentStringObject? {
        val tcfData = deviceStorage.getTCFData()
        val tcString = tcfData.tcString
        if (tcString.isNotBlank()) {
            return ConsentStringObject(
                string = tcString,
                tcfVendorsDisclosedMap = tcfData.vendorsDisclosedMap
            )
        }

        val uspString = deviceStorage.fetchCcpaString()
        if (uspString.isNotBlank()) {
            return ConsentStringObject(
                string = uspString,
            )
        }

        return null
    }

    private fun clearConsentsFromBuffer(consentsData: SaveConsentsData) {
        dispatcher.dispatch {
            val buffer = deviceStorage.getConsentBuffer()
            val newBufferEntries = buffer.entries.filter {
                it.timestampInSeconds != consentsData.timestampInSeconds
            }
            deviceStorage.setConsentBuffer(ConsentsBuffer(newBufferEntries))
        }
    }

    private fun addConsentsToBuffer(consentsData: SaveConsentsData) {
        dispatcher.dispatch {
            val buffer = deviceStorage.getConsentBuffer()
            val entryToAdd = ConsentsBufferEntry(consentsData.timestampInSeconds, consentsData)
            if (!buffer.entries.contains(entryToAdd)) {
                val newBufferEntries = buffer.entries.toMutableList()
                newBufferEntries.add(entryToAdd)
                deviceStorage.setConsentBuffer(ConsentsBuffer(newBufferEntries))
            }
        }
    }

    override fun processConsentsBuffer() {
        dispatcher.dispatch {
            val buffer = deviceStorage.getConsentBuffer()
            buffer.entries.sortedBy { it.timestampInSeconds }.forEach {
                doSaveConsents(it.consents)
            }
        }
    }
}
