package com.usercentrics.sdk.v2.etag.repository

import com.usercentrics.sdk.domain.api.http.HttpResponse
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.v2.etag.cache.EtagCacheStorage
import com.usercentrics.sdk.v2.etag.cache.IEtagCacheStorage

internal abstract class EtagRepository(
    private val logger: UsercentricsLogger,
    private val etagCacheStorage: IEtagCacheStorage
) {

    protected abstract fun etagKey(): String

    fun getApiHeaders(): Map<String, String> {
        val etagValue = getEtagValueForHeader()
        return if (etagValue.isBlank()) {
            emptyMap()
        } else {
            mapOf(etagQueryHeaderEntryKey to etagValue)
        }
    }

    fun getApiBody(response: HttpResponse, cacheMaxAge: Int): String {
        val etagValue = getEtagFromResponse(response.headers)

        if (isEtagValid(response.statusCode)) {
            logger.debug("Valid ETAG cache: key=${etagKey()}")

            if (getCacheControlValueFromEtagFile() == 0L) {
                etagCacheStorage.storeFileAndEtag(EtagHolder(etagKey(), cacheMaxAge, etagValue, response.body))
            }
            return getEtagFile()
        }

        val actualEtagValue = if (etagValue.isNotBlank() && cacheMaxAge != 0) etagValue else dummyEtagKey
        etagCacheStorage.storeFileAndEtag(EtagHolder(etagKey(), cacheMaxAge, actualEtagValue, response.body))

        return response.body
    }

    fun getEtagFile(): String {
        return etagCacheStorage.getStoredFile(etagKey(), getEtagRawFileNameFromCache())
    }

    private fun getEtagRawFileNameFromCache(): String {
        return etagCacheStorage.getRawEtagFileName(etagKey()) ?: ""
    }

    private fun getEtagRawFileNameFromCache2(): String {
        return etagCacheStorage.getRawEtagFileName(etagKey()) ?: ""
    }

    fun getCacheControlValueFromEtagFile(): Long {
        return try {
            val cachedTimestamp = getEtagRawFileNameFromCache().removeSurrounding("\"").split(EtagCacheStorage.cacheControlSeparator)[1]
            cachedTimestamp.toLongOrNull() ?: 0L
        } catch (e: Exception) {
            0L
        }
    }

    private fun getEtagFromResponse(headers: Map<String, String>): String {
        return headers.entries.find { it.key == etagResponseHeaderEntryKey }?.value ?: ""
    }

    private fun isEtagValid(responseStatusCode: Int): Boolean {
        return responseStatusCode == etagValidStatusCode
    }

    private fun getEtagValueForHeader(): String {
        return getEtagRawFileNameFromCache().split(EtagCacheStorage.cacheControlSeparator)[0]
    }

    companion object {
        private const val etagQueryHeaderEntryKey = "If-None-Match"
        private const val etagResponseHeaderEntryKey = "etag"
        private const val etagValidStatusCode = 304
        private const val dummyEtagKey = "xxx"
    }
}
