package com.usercentrics.sdk.v2.language.service

import com.usercentrics.sdk.DeviceLanguage
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.services.deviceStorage.DeviceStorage
import com.usercentrics.sdk.services.tcf.Constants
import com.usercentrics.sdk.v2.language.repository.ILanguageRepository
import com.usercentrics.sdk.v2.location.data.UsercentricsLocation

internal class LanguageService(
    private val languageRepository: ILanguageRepository,
    private val storage: DeviceStorage,
    private val logger: UsercentricsLogger
) : ILanguageService {

    private val platformLanguage = DeviceLanguage()

    override var selectedLanguage: String? = null
        private set
    override var userLocation: UsercentricsLocation? = null
        private set

    override var languagesEtagChanged: Boolean = false

    override suspend fun loadSelectedLanguage(settingsId: String, version: String, defaultLanguage: String, bypassCache: Boolean) {
        val locationAwareResponse = languageRepository.fetchAvailableLanguages(settingsId, version, bypassCache = bypassCache)
        userLocation = locationAwareResponse.location
        languagesEtagChanged = locationAwareResponse.languageEtagChanged

        val availableLanguages = locationAwareResponse.data.map { it.lowercase() }
        selectedLanguage = matchAvailableLanguage(defaultLanguage, availableLanguages)
    }

    private fun matchAvailableLanguage(defaultLanguage: String, availableLanguages: List<String>): String {
        // 1. If language exists in storage, use that
        val storageLanguage = storage.getSettingsLanguage()
        if (storageLanguage.isNotBlank() && storageLanguage in availableLanguages) {
            return storageLanguage
        }

        // 2. If the default language can be found in available languages, use that
        if (defaultLanguage.isNotBlank() && defaultLanguage in availableLanguages) {
            return defaultLanguage
        }

        // 3. If the device language can be found in available languages, use that
        val deviceLanguage = platformLanguage.matchLanguage(platformLanguage.deviceLocale(), availableLanguages)
        if (deviceLanguage?.isNotBlank() == true) {
            logger.debug(deviceLanguageMessage)
            return deviceLanguage
        }

        // 4. Use the first available language
        val firstAvailableLanguage = availableLanguages.firstOrNull()
        if (!firstAvailableLanguage.isNullOrBlank()) {
            logger.debug(firstAvailableLanguageMessage(firstAvailableLanguage))
            return firstAvailableLanguage
        }

        // 5. Use English
        logger.debug(fallbackLanguageMessage)
        return Constants.FALLBACK_LANGUAGE
    }

    companion object {

        private const val fallbackLanguageMessage = "The language has been set to the default one, English."
        private const val deviceLanguageMessage = "The language has been set to the device language."

        private fun firstAvailableLanguageMessage(language: String) = "The language has been set to the first of those available, $language."
    }
}
