package com.usercentrics.sdk.v2.network

import com.usercentrics.sdk.core.application.INetworkStrategy
import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.domain.api.http.HttpResponse
import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.log.UsercentricsLogger
import com.usercentrics.sdk.v2.etag.cache.IEtagCacheStorage
import com.usercentrics.sdk.v2.etag.repository.EtagRepository

internal abstract class NetworkOrchestrator(
    logger: UsercentricsLogger,
    etagCacheStorage: IEtagCacheStorage,
    private val networkStrategy: INetworkStrategy
) : EtagRepository(logger, etagCacheStorage) {

    // FIXME: remove
    fun resolveHttpBody(apiRequest: () -> HttpResponse): String {
        return resolveHttp(apiRequest).body
    }

    suspend fun resolveHttpBody2(bypassCache: Boolean = false, apiRequest: suspend () -> HttpResponse): String {
        return resolveHttp2(bypassCache = bypassCache, apiRequest = apiRequest).body
    }

    fun resolveHttp(apiRequest: () -> HttpResponse): HttpResponse {
        // test corrupt cache
        if (networkStrategy.isOffline()) {
            return httpResponseFromCache()
        }

        val response = apiRequest()
        return processResponse(response)
    }

    suspend fun resolveHttp2(bypassCache: Boolean = false, apiRequest: suspend () -> HttpResponse): HttpResponse {
        return when {
            bypassCache -> processResponse(apiRequest())
            shouldFetchResponseFromCache() -> httpResponseFromCache()
            else -> processResponse(apiRequest())
        }
    }

    fun shouldLoadFromApi(responseCode: Int): Boolean {
        return responseCode == 200
    }

    private fun shouldFetchResponseFromCache(): Boolean {
        if (networkStrategy.isOffline()) {
            return true
        }

        val cachedEtagTimestamp = getCacheControlValueFromEtagFile()
        val currentDateTime = DateTime().timestamp()

        return cachedEtagTimestamp > currentDateTime
    }

    private fun processResponse(response: HttpResponse): HttpResponse {
        val body = when (response.statusCode) {
            200 -> getApiBody(response, response.parseCacheControl())
            304 -> getEtagFile()
            else -> throw UsercentricsException("Invalid Network Response")
        }
        return HttpResponse(headers = response.headers, body = body, statusCode = response.statusCode)
    }

    private fun httpResponseFromCache(): HttpResponse {
        return HttpResponse(headers = mapOf(), body = getEtagFile(), statusCode = 304)
    }
}
