package com.usercentrics.sdk.v2.ruleset.service

import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.models.api.ApiErrors
import com.usercentrics.sdk.v2.location.data.UsercentricsLocation
import com.usercentrics.sdk.v2.location.service.ILocationService
import com.usercentrics.sdk.v2.ruleset.data.GeoRule
import com.usercentrics.sdk.v2.ruleset.data.RuleSet
import com.usercentrics.sdk.v2.ruleset.data.SessionGeoRule
import com.usercentrics.sdk.v2.ruleset.repository.IRuleSetRepository

internal class RuleSetService(
    private val ruleSetRepository: IRuleSetRepository,
    private val locationService: ILocationService,
) : IRuleSetService {

    override suspend fun getActiveSettingsId(id: String): SessionGeoRule {
        val hasCachedLocation = locationService.loadLocation()
        val fetchedRuleSet = ruleSetRepository.fetchRuleSet(id, bypassCache = !hasCachedLocation)

        var location = fetchedRuleSet.second
        if (location.isEmpty()) {
            if (!hasCachedLocation) {
                throw UsercentricsException(ApiErrors.LOCATION_NOT_AVAILABLE)
            }
            location = locationService.location
        }

        val ruleSet = fetchedRuleSet.first
        val targetRule = findRuleForLocation(ruleSet, location)

        val settingsId: String
        var noShow = false

        if (targetRule == null) {
            noShow = ruleSet.defaultRule.noShow
            settingsId = ruleSet.defaultRule.settingsId
        } else {
            settingsId = targetRule.settingsId
        }

        return SessionGeoRule(
            activeSettingsId = settingsId,
            noShow = noShow,
            location = location,
            allSettingsIds = populateAllSettingsIds(ruleSet)
        )
    }

    private fun populateAllSettingsIds(ruleSet: RuleSet): HashSet<String> {
        val result = hashSetOf(ruleSet.defaultRule.settingsId)
        ruleSet.rules.forEach { result.add(it.settingsId) }
        return result
    }

    private fun findRuleForLocation(ruleSet: RuleSet, location: UsercentricsLocation): GeoRule? {
        var regionRule: GeoRule? = null
        var countryRule: GeoRule? = null

        for (rule in ruleSet.rules) {
            if (rule.locations.contains(location.regionCode)) {
                regionRule = rule
                break
            }

            if (rule.locations.contains(location.countryCode)) {
                countryRule = rule
            }
        }
        return regionRule ?: countryRule
    }
}
