package com.usercentrics.sdk.v2.settings.service

import com.usercentrics.sdk.core.settings.SettingsInitializationParameters
import com.usercentrics.sdk.v2.settings.data.*
import com.usercentrics.sdk.v2.settings.repository.IAggregatorRepository
import com.usercentrics.sdk.v2.settings.repository.ISettingsRepository

internal class SettingsService(
    private val settingsRepository: ISettingsRepository,
    private val aggregatorRepository: IAggregatorRepository,
    private val cacheBypassResolver: ICacheBypassResolver,
) : ISettingsService {

    override var settings: NewSettingsData? = null

    override suspend fun loadSettings(settingsInitializationParameters: SettingsInitializationParameters) {
        val settingsValue = getSettings(settingsInitializationParameters)
        val servicesAndCount = getServices(jsonFileLanguage = settingsInitializationParameters.jsonFileLanguage, settings = settingsValue, bypassCache = cacheBypassResolver.shouldBypassCache())

        settings = NewSettingsData(
            data = settingsValue,
            services = servicesAndCount.first,
            servicesCount = servicesAndCount.second
        )
    }

    private suspend fun getSettings(settingsInitializationParameters: SettingsInitializationParameters): UsercentricsSettings {
        val rawSettings = settingsRepository.fetchSettings(settingsInitializationParameters)

        cacheBypassResolver.update(languageEtagChanged = settingsInitializationParameters.languageEtagChanged, settingsEtagChanged = settingsRepository.settingsEtagChanged)

        return removeDeactivatedServices(rawSettings)
    }

    private suspend fun getServices(jsonFileLanguage: String, settings: UsercentricsSettings, bypassCache: Boolean): Pair<List<UsercentricsService>, Int> {
        val categoriesMap = mapCategoriesByCategorySlug(settings.categories)
        val allServicesAndServicesCount = servicesAndSubServicesFromSettings(settings, categoriesMap)

        val allServices = allServicesAndServicesCount.first
        if (allServices.isEmpty()) {
            return Pair(listOf(), 0)
        }

        val servicesCount = allServicesAndServicesCount.second
        val aggregatorServices = aggregatorRepository.fetchServices(jsonFileLanguage, allServices, bypassCache)
        return Pair(aggregateServicesByCategory(settings.consentTemplates, aggregatorServices, categoriesMap), servicesCount)
    }

    private fun mapCategoriesByCategorySlug(categories: List<UsercentricsCategory>?): Map<String, UsercentricsCategory> {
        if (categories == null) {
            return emptyMap()
        }
        return categories.associateBy { it.categorySlug }
    }

    private fun aggregateServicesByCategory(
        consentTemplates: List<ServiceConsentTemplate>,
        aggregatorServices: List<UsercentricsService>,
        categoriesByCategoriesSlugsMap: Map<String, UsercentricsCategory>
    ): List<UsercentricsService> {
        val services = mutableListOf<UsercentricsService>()

        aggregatorServices.map { aggregatorService ->
            val consentTemplateMatchedWithAggregatorService = consentTemplates.find { aggregatorService.templateId == it.templateId }
            val consentTemplateSlug = consentTemplateMatchedWithAggregatorService?.categorySlug
            val categoryMatchedByConsentTemplateSlug = categoriesByCategoriesSlugsMap[consentTemplateSlug]

            if (consentTemplateMatchedWithAggregatorService != null && categoryMatchedByConsentTemplateSlug != null) {
                services.add(updateService(aggregatorService, consentTemplateMatchedWithAggregatorService, categoryMatchedByConsentTemplateSlug))
            }
        }
        return services
    }

    private fun updateService(aggregatorService: UsercentricsService, consentTemplate: ServiceConsentTemplate, category: UsercentricsCategory): UsercentricsService {
        return aggregatorService.copy(
            legalBasisList = if (consentTemplate.legalBasisList?.isEmpty() == false) {
                consentTemplate.legalBasisList
            } else {
                aggregatorService.legalBasisList
            },
            categorySlug = category.categorySlug,
            isDeactivated = consentTemplate.isDeactivated,
            isAutoUpdateAllowed = consentTemplate.isAutoUpdateAllowed,
            disableLegalBasis = consentTemplate.disableLegalBasis,
            isEssential = category.isEssential,
            isHidden = category.isHidden || consentTemplate.isHidden
        )
    }

    private fun removeDeactivatedServices(settings: UsercentricsSettings): UsercentricsSettings {
        val activeServices = settings.consentTemplates.filterNot { it.isDeactivated == true }
        return settings.copy(consentTemplates = activeServices)
    }

    // TODO: we can improve this routine
    // avoid too many iterations on consentTemplate (see mergeServicesWithConsentTemplate)
    // sortedBy in the end could be removed, if we have a list that sorts itself on every addition
    private fun servicesAndSubServicesFromSettings(settings: UsercentricsSettings, categoriesMap: Map<String, UsercentricsCategory>): Pair<List<BasicConsentTemplate>, Int> {
        val result = mutableListOf<BasicConsentTemplate>()
        var servicesCount = 0
        settings.consentTemplates.forEach { service ->
            if (!categoriesMap.containsKey(service.categorySlug)) {
                return@forEach
            }

            result.add(basicConsentTemplate(service))
            service.subConsents.forEach { result.add(basicConsentTemplate(it)) }

            servicesCount++
        }
        return Pair(result.sortedBy { it.templateId }, servicesCount)
    }

    private fun basicConsentTemplate(service: ConsentTemplate): BasicConsentTemplate {
        return BasicConsentTemplate(templateId = service.templateId, version = service.version)
    }
}
