package com.usercentrics.tcf.core

import com.usercentrics.tcf.core.encoder.*
import com.usercentrics.tcf.core.encoder.field.IntEncoder
import com.usercentrics.tcf.core.encoder.sequence.SegmentSequence
import com.usercentrics.tcf.core.encoder.sequence.SequenceVersionMapType
import com.usercentrics.tcf.core.errors.DecodingError
import com.usercentrics.tcf.core.errors.EncodingError
import com.usercentrics.tcf.core.model.Segment
import com.usercentrics.tcf.core.model.SegmentIDs

/**
 * Main class for encoding and decoding a
 * TCF Transparency and Consent String
 */
class TCString {

    companion object {

        /**
         * encodes a model into a TCString
         *
         * @param {TCModel} tcModel - model to convert into encoded string
         * @param {EncodingOptions} options - for encoding options other than default
         * @return {string} - base64url encoded Transparency and Consent String
         */
        internal fun encode(tcModel: TCModel): String {
            val internalTCModel = SemanticPreEncoder.process(tcModel)
            val version = internalTCModel.getVersion()
            if (version != 2) {
                throw EncodingError("Unsupported TCF version: $version")
            }

            var out = ""

            /**
             * If they pass in a special segment sequence.
             */
            val sequence = (SegmentSequence(internalTCModel).two as SequenceVersionMapType.List).value

            sequence.forEachIndexed { index: Int, segment: Segment ->
                var dotMaybe = ""

                if (index < sequence.size - 1) {
                    dotMaybe = "."
                }

                out += SegmentEncoder.encode(internalTCModel, segment) + dotMaybe
            }

            return out
        }

        /**
         * Decodes a string into a TCModel
         *
         * @param {string} encodedTCString - base64url encoded Transparency and
         * Consent String to decode - can also be a single or group of segments of
         * the string
         * @param {string} [tcModel] - model to enhance with the information.  If
         * none is passed a new instance of TCModel will be created.
         * @return {TCModel} - Returns populated TCModel
         */
        internal fun decode(encodedTCString: String, tcModel: TCModel): TCModel {
            val version = tcModel.getVersion()
            if (version != 2) {
                throw DecodingError("Unable TCF String with version $version")
            }

            val segments = encodedTCString.split(".")
            val len = segments.size
            val tempTcModel: TCModel = tcModel

            var retrTCModel: TCModel? = null

            for (i in 0 until len) {
                val segString: String = segments[i]

                /**
                 * first char will contain 6 bits, we only need the first 3. In version 1
                 * and 2 of the TC string there is no segment type for the CORE string.
                 * Instead the first 6 bits are reserved for the encoding version, but
                 * because we're only on a maximum of encoding version 2 the first 3 bits
                 * in the core segment will evaluate to 0.
                 */
                val firstChar: String = Base64Url.decode(segString[0].toString())
                val segTypeBits: String = firstChar.substring(0, BitLength.segmentType.integer)
                val segment = SegmentIDs.ID_TO_KEY[IntEncoder.decode(segTypeBits, BitLength.segmentType.integer).toInt()].type

                retrTCModel = SegmentEncoder.decode(segString, tempTcModel, segment)
            }

            if (retrTCModel != null) {
                return retrTCModel
            }

            throw DecodingError("Unable to decode given TCModel")
        }
    }
}
