package com.usercentrics.tcf.core.encoder

import com.usercentrics.tcf.core.errors.DecodingError
import com.usercentrics.tcf.core.errors.EncodingError

@Suppress("SpellCheckingInspection")
internal class Base64Url {

    /**
     * Base 64 URL character set.  Different from standard Base64 char set
     * in that '+' and '/' are replaced with '-' and '_'.
     */
    companion object {
        private const val DICT = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"

        // "A" to 0, "B" to 1, "C" to 2, "D" to 3, "E" to 4, "F" to 5, "G" to 6, "H" to 7 ...
        private val REVERSE_DICT: Map<String, Int> by lazy {
            mutableMapOf<String, Int>().apply {
                DICT.forEachIndexed { index, c ->
                    this[c.toString()] = index
                }
            }
        }

        /**
         * log2(64) = 6
         */
        private const val BASIS = 6
        private const val LCM = 24

        /**
         * encodes an arbitrary-length bitfield string into base64url
         *
         * @static
         * @param {string} str - arbitrary-length bitfield string to be encoded to base64url
         * @return {string} - base64url encoded result
         */
        fun encode(str: String): String {
            var tempStr = str
            val binaryRegex = Regex("[0-1]+")

            /**
             * Pad the end of the string to the least common multiple of 6 (basis for
             * base64) and 8 (one byte)
             */
            val padding = tempStr.length % LCM
            if (padding > 0) {
                tempStr += "0".repeat(LCM - padding)
            }

            var result = ""
            for (i in tempStr.indices step BASIS) {
                val subString = tempStr.substring(i, i + BASIS)
                // Verify that the substring is a binary string
                if (!binaryRegex.matches(subString)) {
                    throw EncodingError("Invalid bitField")
                }

                val baseTwoInt = subString.toInt(2)
                result += DICT[baseTwoInt]
            }
            return result
        }


        /**
         * decodes a base64url encoded bitfield string
         *
         * @static
         * @param {string} str - base64url encoded bitfield string to be decoded
         * @return {string} - bitfield string
         */
        fun decode(str: String): String {
            var result = ""
            try {
                for (i in str.indices) {
                    /**
                     * index the binary value of the character from out reverse map
                     */
                    val fetchedChar = REVERSE_DICT[str[i].toString()] ?: throw DecodingError("Invalid value on index $i")

                    val strBits = fetchedChar.toString(2)
                    /**
                     * Since a bit string converted to an integer on encoding will lose
                     * leading zeros – pad to the left for those missing leading zeros
                     */
                    result += "0".repeat(BASIS - strBits.length) + strBits
                }
            } catch (ex: Exception) {
                throw DecodingError("Invalid encoded Base64URL string")
            }
            return result
        }
    }
}
