package com.usercentrics.tcf.core.encoder

import com.usercentrics.tcf.core.*
import com.usercentrics.tcf.core.encoder.field.*
import com.usercentrics.tcf.core.encoder.sequence.SegmentSequence
import com.usercentrics.tcf.core.encoder.sequence.SequenceVersionMapType
import com.usercentrics.tcf.core.errors.EncodingError
import com.usercentrics.tcf.core.model.Segment

internal class TCFKeysEncoder(
    private val tcModel: TCModel,
    private val tcString: String,
    private val gdprApplies: Int
) {
    // Stored Keys
    private var cmpIdEncoded: Int? = null
    private var cmpVersionEncoded: Int? = null
    private var policyVersionEncoded: Int? = null
    private var publisherCountryCodeEncoded: String = publisherCountryCodeDefault
    private var purposeOneTreatmentEncoded: Int? = null
    private var useNonStandardStacksEncoded: Int? = null
    private var vendorConsentsEncoded: String = ""
    private var vendorLegitimateInterestsEncoded: String = ""
    private var purposeConsentsEncoded: String = ""
    private var purposeLegitimateInterestsEncoded: String = ""
    private var specialFeatureOptinsEncoded: String = ""
    private var publisherRestrictionsEncoded: Map<Int, String> = emptyMap()
    private var publisherConsentsEncoded: String = ""
    private var publisherLegitimateInterestsEncoded: String = ""
    private var publisherCustomConsentsEncoded: String = ""
    private var publisherCustomLegitimateInterestsEncoded: String = ""
    private val enableAdvertiserConsentMode: Int = 1

    // Non Stored Keys
    private var versionEncoded: Int? = null
    private var numCustomPurposesEncoded: String = ""
    private var consentScreenEncoded: String = ""
    private var vendorListVersionEncoded: String = ""
    private var segmentTypeEncoded: String = ""
    private var createdEncoded: String = ""
    private var lastUpdatedEncoded: String = ""
    private var consentLanguageEncoded: String = ""
    private var isServiceSpecificEncoded: String = ""
    private var vendorsDisclosedEncoded: String = ""
    private var vendorsAllowedEncoded: String = ""

    fun encode(): TCFKeys {
        val internalTCModel = SemanticPreEncoder.process(tcModel)
        val version = internalTCModel.getVersion()
        if (version != 2) {
            throw EncodingError("Error encoding TCF String. Invalid version: $version")
        }

        // If they pass in a special segment sequence.
        val sequence = (SegmentSequence(internalTCModel).two as SequenceVersionMapType.List).value

        sequence.forEach { segment: Segment ->
            encodeSegment(segment)
        }

        return TCFKeys(
            IABTCF_TCString = tcString,
            IABTCF_gdprApplies = gdprApplies,
            IABTCF_CmpSdkID = cmpIdEncoded,
            IABTCF_CmpSdkVersion = cmpVersionEncoded,
            IABTCF_PolicyVersion = policyVersionEncoded,
            IABTCF_PublisherCC = publisherCountryCodeEncoded,
            IABTCF_PurposeOneTreatment = purposeOneTreatmentEncoded,
            IABTCF_UseNonStandardStacks = useNonStandardStacksEncoded,
            IABTCF_VendorConsents = vendorConsentsEncoded,
            IABTCF_VendorLegitimateInterests = vendorLegitimateInterestsEncoded,
            IABTCF_PurposeConsents = purposeConsentsEncoded,
            IABTCF_PurposeLegitimateInterests = purposeLegitimateInterestsEncoded,
            IABTCF_SpecialFeaturesOptIns = specialFeatureOptinsEncoded,
            IABTCF_PublisherRestrictions = publisherRestrictionsEncoded,
            IABTCF_PublisherConsent = publisherConsentsEncoded,
            IABTCF_PublisherLegitimateInterests = publisherLegitimateInterestsEncoded,
            IABTCF_PublisherCustomPurposesConsents = publisherCustomConsentsEncoded,
            IABTCF_PublisherCustomPurposesLegitimateInterests = publisherCustomLegitimateInterestsEncoded,
            IABTCF_EnableAdvertiserConsentMode = enableAdvertiserConsentMode,
            IABTCF_DisclosedVendors = vendorsDisclosedEncoded
        )
    }

    private fun encodeSegment(segment: Segment) {
        getSequenceForSegment(segment)?.forEach { key ->
            val value: TCModelPropType = tcModel.getFieldByName(key)
            val bitLength = BitLength.getByName(key)
            var numBits = 0
            if (bitLength == null) {
                if (SegmentEncoder.isPublisherCustom(key)) {
                    /**
                     * publisherCustom[Consents | LegitimateInterests] are an edge case
                     * because they are of variable length. The length is defined in a
                     * separate field named numCustomPurposes.
                     */
                    val numCustomPurposes = tcModel.getNumCustomPurposes()
                    if (numCustomPurposes is StringOrNumber.Int) {
                        numBits = numCustomPurposes.value
                    }
                    if (numCustomPurposes is StringOrNumber.String) {
                        numBits = numCustomPurposes.value.toInt()
                    }
                }
            } else {
                numBits = bitLength.integer
            }
            try {
                when (key) {
                    "version", "numCustomPurposes" -> {
                        value as TCModelPropType.Int
                        if (key == "version") {
                            versionEncoded = value.value
                        } else {
                            numCustomPurposesEncoded =
                                IntEncoder.encode(StringOrNumber.Int(value.value), numBits)
                        }
                    }

                    "cmpId",
                    "cmpVersion",
                    "consentScreen",
                    "vendorListVersion",
                    "policyVersion" -> {
                        value as TCModelPropType.StringOrNumber
                        val internalValue = value.value
                        internalValue as StringOrNumber.Int
                        if (key == "cmpId") {
                            cmpIdEncoded = internalValue.value
                        }
                        if (key == "cmpVersion") {
                            cmpVersionEncoded = internalValue.value
                        }
                        if (key == "consentScreen") {
                            consentScreenEncoded =
                                IntEncoder.encode(StringOrNumber.Int(internalValue.value), numBits)
                        }
                        if (key == "vendorListVersion") {
                            vendorListVersionEncoded =
                                IntEncoder.encode(StringOrNumber.Int(internalValue.value), numBits)
                        }
                        if (key == "policyVersion") {
                            policyVersionEncoded = internalValue.value
                        }
                    }

                    "segmentType" -> {
                        value as TCModelPropType.StringOrNumber
                        segmentTypeEncoded = IntEncoder.encode(value.value, numBits)
                    }

                    "created", "lastUpdated" -> {
                        value as TCModelPropType.Date
                        if (key == "created") {
                            createdEncoded = DateEncoder.encode(value.value!!, numBits)
                        }
                        if (key == "lastUpdated") {
                            lastUpdatedEncoded = DateEncoder.encode(value.value!!, numBits)
                        }
                    }

                    "consentLanguage", "publisherCountryCode" -> {
                        value as TCModelPropType.String
                        if (key == "consentLanguage") {
                            consentLanguageEncoded = value.value
                        }
                        if (key == "publisherCountryCode") {
                            publisherCountryCodeEncoded = value.value
                        }
                    }

                    "isServiceSpecific",
                    "useNonStandardStacks",
                    "purposeOneTreatment" -> {
                        value as TCModelPropType.Boolean
                        if (key == "isServiceSpecific") {
                            isServiceSpecificEncoded = BooleanEncoder.encode(value.value)
                        }
                        if (key == "useNonStandardStacks") {
                            useNonStandardStacksEncoded = BooleanEncoder.encode(value.value).toInt()
                        }
                        if (key == "purposeOneTreatment") {
                            purposeOneTreatmentEncoded = BooleanEncoder.encode(value.value).toInt()
                        }
                    }

                    "specialFeatureOptins" -> {
                        value as TCModelPropType.Vector
                        specialFeatureOptinsEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "purposeConsents" -> {
                        value as TCModelPropType.Vector
                        purposeConsentsEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "purposeLegitimateInterests" -> {
                        value as TCModelPropType.Vector
                        purposeLegitimateInterestsEncoded =
                            FixedVectorEncoder.encode(value.value, null)
                    }

                    "publisherConsents" -> {
                        value as TCModelPropType.Vector
                        publisherConsentsEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "publisherLegitimateInterests" -> {
                        value as TCModelPropType.Vector
                        publisherLegitimateInterestsEncoded =
                            FixedVectorEncoder.encode(value.value, null)
                    }

                    "publisherCustomConsents" -> {
                        value as TCModelPropType.Vector
                        publisherCustomConsentsEncoded =
                            FixedVectorEncoder.encode(value.value, null)
                    }

                    "publisherCustomLegitimateInterests" -> {
                        value as TCModelPropType.Vector
                        publisherCustomLegitimateInterestsEncoded =
                            FixedVectorEncoder.encode(value.value, null)
                    }

                    "vendorConsents" -> {
                        value as TCModelPropType.Vector
                        vendorConsentsEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "vendorLegitimateInterests" -> {
                        value as TCModelPropType.Vector
                        vendorLegitimateInterestsEncoded =
                            FixedVectorEncoder.encode(value.value, null)
                    }

                    "vendorsDisclosed" -> {
                        value as TCModelPropType.Vector
                        vendorsDisclosedEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "vendorsAllowed" -> {
                        value as TCModelPropType.Vector
                        vendorsAllowedEncoded = FixedVectorEncoder.encode(value.value, null)
                    }

                    "publisherRestrictions" -> {
                        value as TCModelPropType.PurposeRestrictionVector
                        publisherRestrictionsEncoded = PublisherRestrictionsEncoder.encode(value.value)
                    }

                    else -> {
                        throw EncodingError("Error encoding $segment->$key, value: $value")
                    }
                }
            } catch (error: Throwable) {
                throw EncodingError("Error encoding $segment->$key: ${error.message}")
            }
        }
    }

    private fun getSequenceForSegment(segment: Segment): List<String>? {
        return (SegmentEncoder.fieldSequence.two as SequenceVersionMapType.SVMItemMap).map[segment]
    }
}
