package com.usercentrics.tcf.core.encoder.field

import com.usercentrics.tcf.core.StringOrNumber
import com.usercentrics.tcf.core.encoder.BitLength
import com.usercentrics.tcf.core.errors.DecodingError
import com.usercentrics.tcf.core.model.PurposeRestriction
import com.usercentrics.tcf.core.model.PurposeRestrictionVector
import com.usercentrics.tcf.core.model.RestrictionType

internal class PurposeRestrictionVectorEncoder {

    companion object {

        fun encode(prVector: PurposeRestrictionVector): String {
            // start with the number of restrictions
            var bitString = IntEncoder.encode(StringOrNumber.Int(prVector.getNumRestrictions()), BitLength.numRestrictions.integer)

            val gvl = prVector.getGVL()

            // if the vector is empty we'll just return a string with just the numRestrictions being 0
            if (prVector.isEmpty() || gvl == null) {
                return bitString
            }

            fun nextGvlVendor(vendorId: Int, lastVendorId: Int): Int {
                var nextId = vendorId + 1
                while (nextId <= lastVendorId) {
                    if (gvl.vendorIds?.contains(nextId) == true) {
                        return nextId
                    }
                    nextId++
                }
                return vendorId
            }

            // create each restriction group
            prVector.getRestrictions(null).forEach { purpRestriction ->

                // every restriction group has the purposeId and the restrictionType
                bitString += IntEncoder.encode(StringOrNumber.Int(purpRestriction.getPurposeId()!!), BitLength.purposeId.integer)
                bitString += IntEncoder.encode(StringOrNumber.Int(purpRestriction.restrictionType.value), BitLength.restrictionType.integer)

                // now get all the vendors under that restriction
                val vendors: List<Int> = prVector.getVendors(purpRestriction)
                val len: Int = vendors.size

                /**
                 * numEntries comes first so we will have to keep a counter and the do
                 * the encoding at the end
                 */
                var numEntries = 0
                var startId = 0
                var rangeField = ""

                for (i in 0 until len) {
                    val vendorId: Int = vendors[i]
                    if (startId == 0) {
                        numEntries += 1
                        startId = vendorId
                    }

                    // either end of the loop or there's a gap greater than 1 number
                    if (i == len - 1 || vendors[i + 1] > nextGvlVendor(vendorId, vendors[len - 1])) {
                        // it's a range entry if we've got something other than the start Id
                        val isRange = vendorId != startId

                        // 0 means single 1 means range
                        rangeField += BooleanEncoder.encode(isRange)
                        rangeField += IntEncoder.encode(StringOrNumber.Int(startId), BitLength.vendorId.integer)

                        if (isRange) {
                            rangeField += IntEncoder.encode(StringOrNumber.Int(vendorId), BitLength.vendorId.integer)
                        }
                        // reset the startId so we grab the next id in the list
                        startId = 0
                    }
                }

                /**
                 * now that  the range encoding is built, encode the number of ranges
                 * and then append the range field to the bitString.
                 */
                bitString += IntEncoder.encode(StringOrNumber.Int(numEntries), BitLength.numEntries.integer)
                bitString += rangeField
            }
            return bitString
        }

        fun decode(encodedString: String): PurposeRestrictionVector {
            var index = 0
            val vector = PurposeRestrictionVector()
            val numRestrictions: Int = IntEncoder.decode(encodedString.substring(index, index + BitLength.numRestrictions.integer), BitLength.numRestrictions.integer).toInt()

            index += BitLength.numRestrictions.integer

            for (i in 0 until numRestrictions) {
                // First is purpose Id
                val purposeId = IntEncoder.decode(
                    encodedString.substring(index, index + BitLength.purposeId.integer),
                    BitLength.purposeId.integer
                ).toInt()
                index += BitLength.purposeId.integer

                // Second Restriction Type
                val restrictionType = IntEncoder.decode(
                    encodedString.substring(index, index + BitLength.restrictionType.integer),
                    BitLength.restrictionType.integer
                ).toInt()
                index += BitLength.restrictionType.integer

                val purposeRestriction = PurposeRestriction(
                    purposeId,
                    RestrictionType.getRestrictionTypeByValue(restrictionType)
                )

                // Num Entries (number of vendors)
                val numEntries: Int = IntEncoder.decode(
                    encodedString.substring(index, index + BitLength.numEntries.integer),
                    BitLength.numEntries.integer
                ).toInt()
                index += BitLength.numEntries.integer

                for (j in 0 until numEntries) {
                    val isARange: Boolean = BooleanEncoder.decode(encodedString.substring(index, index + BitLength.anyBoolean.integer))
                    index += BitLength.anyBoolean.integer

                    val startOrOnlyVendorId: Int = IntEncoder.decode(
                        encodedString.substring(index, index + BitLength.vendorId.integer),
                        BitLength.vendorId.integer
                    ).toInt()
                    index += BitLength.vendorId.integer

                    if (isARange) {
                        val endVendorId: Int = IntEncoder.decode(encodedString.substring(index, index + BitLength.vendorId.integer), BitLength.vendorId.integer).toInt()
                        index += BitLength.vendorId.integer

                        if (endVendorId < startOrOnlyVendorId) {
                            throw DecodingError("Invalid RangeEntry: endVendorId $endVendorId is less than $startOrOnlyVendorId")
                        }

                        for (k in startOrOnlyVendorId..endVendorId) {
                            vector.add(k, purposeRestriction)
                        }
                    } else {
                        vector.add(startOrOnlyVendorId, purposeRestriction)
                    }
                }
            }

            vector.bitLength = index
            return vector
        }
    }
}
