package com.usercentrics.tcf.core.encoder.sequence

import com.usercentrics.tcf.core.TCModel
import com.usercentrics.tcf.core.model.Segment

/**
 * Defines the order and inclusion of segments in the TC String encoding.
 *
 * From TCF v2.3 onward, the "Disclosed Vendors" segment (VENDORS_DISCLOSED)
 * must always be included — regardless of whether the TC String is service-specific
 * or if OOB (Out Of Band) processing is supported.
 */
internal class SegmentSequence(tcModel: TCModel) : SequenceVersionMap() {

    /**
     * Version 2 of the TCF sequence starts with the CORE segment by default.
     * Additional segments may be appended based on model configuration.
     */
    override var two: SequenceVersionMapType = SequenceVersionMapType.List(listOf(Segment.CORE))

    init {
        val tempSeqVersionMap = (this.two as SequenceVersionMapType.List).value.toMutableList()

        /**
         * [TCF 2.3] The "Disclosed Vendors" segment (VENDORS_DISCLOSED)
         * is now mandatory for all TC strings.
         *
         * In TCF 2.2 and earlier versions, this segment was optional and typically
         * only included when the TC string was meant for saving globally (supportOOB = true).
         *
         * However, starting from TCF 2.3, the "Disclosed Vendors" segment must always
         * be included — regardless of whether the TC String is global or service-specific,
         * or whether Out Of Band (OOB) consent is supported.
         */
        tempSeqVersionMap.add(Segment.VENDORS_DISCLOSED)

        if (tcModel.getIsServiceSpecific()) {
            /**
             * If it's service-specific only, then the Publisher TC String can be
             * stored in the cookie and would be transmitted even if it's not meant
             * for storage. Therefore, it's included regardless of whether the TC
             * string is for saving or just for surfacing through the CMP API.
             */
            tempSeqVersionMap.add(Segment.PUBLISHER_TC)
        }

        this.two = SequenceVersionMapType.List(tempSeqVersionMap.toList())
    }
}
