package com.usercentrics.tcf.core.model

import com.usercentrics.tcf.core.GVL
import kotlin.math.max

internal data class PurposeRestrictionVector(
    /**
     * if this originated from an encoded string we'll need a place to store the
     * bit length; it can be set and got from here
     */
    var bitLength: Int = 0,

    /**
     * a map indexed by a string which will be a 'hash' of the purpose and
     * restriction type.
     *
     * Using a BST to keep vendors in a sorted order for encoding later
     */
    val map: MutableMap<String, SortedSet<Int>> = mutableMapOf()
) {

    private val initTCModelRestrictPurposeToLegalBasisCache: MutableSet<String> = mutableSetOf()

    @Suppress("PrivatePropertyName")
    private var gvl_: GVL? = null

    private fun has(hash: String): Boolean {
        return this.map.contains(hash)
    }

    /**
     * add - adds a given Vendor ID under a given Purpose Restriction
     *
     * @param {number} vendorId
     * @param {PurposeRestriction} purposeRestriction
     * @return {void}
     */
    fun add(vendorId: Int, purposeRestriction: PurposeRestriction) {
        val hash: String = purposeRestriction.getHash()

        if (!this.has(hash)) {
            map[hash] = SortedSet<Int>().apply { add(vendorId) }
            this.bitLength = 0
        } else {
            map[hash]?.add(vendorId)
        }
    }

    /**
     * getVendors - returns array of vendor ids optionally narrowed by a given
     * Purpose Restriction.  If no purpose restriction is passed then all vendor
     * ids will be returned.  One can expect this result to be a unique set of
     * ids no duplicates.
     *
     * @param {PurposeRestriction} [purposeRestriction] - optionally passed to
     * get only Vendor IDs restricted under the given Purpose Restriction
     * @return {number[]} - Unique ID set of vendors
     */
    fun getVendors(purposeRestriction: PurposeRestriction?): List<Int> {
        var vendorIds: List<Int> = listOf()

        if (purposeRestriction != null) {
            val hash: String = purposeRestriction.getHash()
            if (this.has(hash)) {
                vendorIds = (this.map[hash] as SortedSet).get().toList()
            }
        } else {
            val vendorSet = mutableSetOf<Int>()
            this.map.forEach {
                it.value.get().forEach { vendorId ->
                    vendorSet.add(vendorId)
                }
            }
            vendorIds = vendorSet.toList()
        }
        return vendorIds
    }

    fun getRestrictionType(vendorId: Int, purposeId: Int): RestrictionType? {
        var rType: RestrictionType? = null

        this.getRestrictions(vendorId).forEach { purposeRestriction ->
            if (purposeRestriction.getPurposeId() == purposeId) {
                if (rType == null || rType?.ordinal!! > purposeRestriction.restrictionType.ordinal) {
                    rType = purposeRestriction.restrictionType
                }
            }
        }
        return rType
    }

    fun initTCModelRestrictPurposeToLegalBasis(restrictionsHashes: Set<String>) {
        val vendors = this.gvl_?.vendorIds ?: return

        restrictionsHashes.forEach { hash ->
            // Performance improvement: avoid to process again a hash that has been already processed by this method (not the other add method)
            val alreadyProcessed = initTCModelRestrictPurposeToLegalBasisCache.contains(hash)
            if (!alreadyProcessed) {
                initTCModelRestrictPurposeToLegalBasisCache.add(hash)
                this.map[hash] = SortedSet<Int>().apply {
                    (vendors).forEach { add(it) }
                }
                this.bitLength = 0
            }
        }
    }

    /**
     * restrictPurposeToLegalBasis - adds all Vendors under a given Purpose Restriction
     *
     * @param {PurposeRestriction} purposeRestriction
     * @return {void}
     */
    fun initTCModelRestrictPurposeToLegalBasis(purposeRestriction: PurposeRestriction) {
        val vendors = this.gvl_?.vendorIds ?: return
        val hash = purposeRestriction.getHash()

        // Performance improvement: avoid to process again a hash that has been already processed by this method (not the other add method)
        val alreadyProcessed = initTCModelRestrictPurposeToLegalBasisCache.contains(hash)
        if (!alreadyProcessed) {
            initTCModelRestrictPurposeToLegalBasisCache.add(hash)
            this.map[hash] = SortedSet<Int>().apply {
                (vendors).forEach { add(it) }
            }
            this.bitLength = 0
        }
    }

    /**
     * getMaxVendorId - gets the Maximum Vendor ID regardless of Purpose
     * Restriction
     *
     * @return {number} - maximum Vendor ID
     */
    fun getMaxVendorId(): Int {
        var retr = 0

        this.map.forEach {
            val bst = it.value
            val bstMax = bst.max()
            if (bstMax != null) {
                retr = max(bstMax, retr)
            }
        }
        return retr
    }

    @Suppress("MemberVisibilityCanBePrivate")
    fun getRestrictions(vendorId: Int? = null): List<PurposeRestriction> {
        val result = mutableListOf<PurposeRestriction>()

        this.map.forEach {
            val bst = it.value
            val hash = it.key
            if (vendorId != null) {
                if (bst.contains(vendorId)) {
                    result.add(PurposeRestriction.unHash(hash))
                }
            } else {
                result.add(PurposeRestriction.unHash(hash))
            }
        }
        return result
    }

    /**
     * Essential for being able to determine whether we can actually set a
     * purpose restriction since they have to have a flexible legal basis
     *
     * @param {GVL} value - the GVL instance
     */
    internal fun setGvl(value: GVL): PurposeRestrictionVector {
        if (this.gvl_ != null) {
            return this
        }

        this.gvl_ = value
        return this
    }

    /**
     * gvl returns local copy of the GVL these restrictions apply to
     *
     * @return {GVL}
     */
    internal fun getGVL(): GVL? {
        return this.gvl_
    }

    /**
     * isEmpty - whether or not this vector has any restrictions in it
     *
     * @return {boolean}
     */
    fun isEmpty(): Boolean {
        return this.map.isEmpty()

    }

    /**
     * numRestrictions - returns the number of Purpose Restrictions.
     *
     * @return {number}
     */
    fun getNumRestrictions(): Int {
        return this.map.size
    }
}
