package com.usercentrics.sdk

import android.graphics.Typeface
import android.view.Gravity
import androidx.annotation.ColorInt
import com.usercentrics.sdk.models.settings.PredefinedUIVariant
import com.usercentrics.sdk.v2.settings.data.FirstLayerLogoPosition
import com.usercentrics.sdk.v2.settings.data.FirstLayerMobileVariant

data class BannerSettings(
    val generalStyleSettings: GeneralStyleSettings? = null,
    val firstLayerStyleSettings: FirstLayerStyleSettings? = null,
    val secondLayerStyleSettings: SecondLayerStyleSettings? = null,
    val variantName: String? = null,
)

data class GeneralStyleSettings(
    @ColorInt val textColor: Int? = null,
    @ColorInt val layerBackgroundColor: Int? = null,
    @ColorInt val layerBackgroundSecondaryColor: Int? = null,
    @ColorInt val linkColor: Int? = null,
    @ColorInt val tabColor: Int? = null,
    @ColorInt val bordersColor: Int? = null,
    val toggleStyleSettings: ToggleStyleSettings? = null,
    val font: BannerFont? = null,
    val logo: UsercentricsImage? = null,
    val links: LegalLinksSettings? = null,
    val disableSystemBackButton: Boolean? = null,
    @ColorInt val statusBarColor: Int? = null,
    val windowFullscreen: Boolean? = null,
)

data class ToggleStyleSettings(
    @ColorInt val activeBackgroundColor: Int? = null,
    @ColorInt val inactiveBackgroundColor: Int? = null,
    @ColorInt val disabledBackgroundColor: Int? = null,
    @ColorInt val activeThumbColor: Int? = null,
    @ColorInt val inactiveThumbColor: Int? = null,
    @ColorInt val disabledThumbColor: Int? = null,
)

enum class LegalLinksSettings {
    FIRST_LAYER_ONLY,
    SECOND_LAYER_ONLY,
    BOTH,
    HIDDEN,
}

data class FirstLayerStyleSettings(
    val layout: UsercentricsLayout? = null,
    val headerImage: HeaderImageSettings? = null,
    val title: TitleSettings? = null,
    val message: MessageSettings? = null,
    val buttonLayout: ButtonLayout? = null,
    @ColorInt val backgroundColor: Int? = null,
    val cornerRadius: Int? = null,
    @ColorInt val overlayColor: Int? = null,
)

sealed class UsercentricsLayout {

    object Sheet : UsercentricsLayout()

    data class Popup(
        val position: PopupPosition,
        val horizontalMarginInDp: Float? = null,
        val verticalMarginInDp: Float? = null,
    ) : UsercentricsLayout()

    object Full : UsercentricsLayout()

    internal fun predefinedUIVariant(): PredefinedUIVariant {
        return when (this) {
            is Sheet -> PredefinedUIVariant.SHEET
            is Full -> PredefinedUIVariant.FULL
            is Popup -> {
                if (position == PopupPosition.CENTER) PredefinedUIVariant.POPUP_CENTER else PredefinedUIVariant.POPUP_BOTTOM
            }
        }
    }
}

internal fun FirstLayerMobileVariant.toUsercentricsLayout(): UsercentricsLayout = when (this) {
    FirstLayerMobileVariant.SHEET -> UsercentricsLayout.Sheet
    FirstLayerMobileVariant.FULL -> UsercentricsLayout.Full
    FirstLayerMobileVariant.POPUP_BOTTOM -> UsercentricsLayout.Popup(PopupPosition.BOTTOM)
    FirstLayerMobileVariant.POPUP_CENTER -> UsercentricsLayout.Popup(PopupPosition.CENTER)
}

enum class PopupPosition {
    BOTTOM,
    CENTER
}

sealed class HeaderImageSettings {
    data class ExtendedLogoSettings(val image: UsercentricsImage) : HeaderImageSettings()
    data class LogoSettings(
        val image: UsercentricsImage,
        val alignment: SectionAlignment? = null,
        val heightInDp: Float? = null
    ) : HeaderImageSettings()

    object Hidden : HeaderImageSettings()
}

data class TitleSettings(
    val font: Typeface? = null,
    val textSizeInSp: Float? = null,
    val alignment: SectionAlignment? = null,
    @ColorInt val textColor: Int? = null,
)

data class MessageSettings(
    val font: Typeface? = null,
    val textSizeInSp: Float? = null,
    val alignment: SectionAlignment? = null,
    @ColorInt val textColor: Int? = null,
    @ColorInt val linkTextColor: Int? = null,
    val underlineLink: Boolean? = null,
)

enum class SectionAlignment {
    START,
    CENTER,
    END;

    companion object {
        internal fun from(firstLayerLogoPosition: FirstLayerLogoPosition?): SectionAlignment? {
            return when (firstLayerLogoPosition) {
                FirstLayerLogoPosition.LEFT -> START
                FirstLayerLogoPosition.CENTER -> CENTER
                FirstLayerLogoPosition.RIGHT -> END
                null -> null
            }
        }

        internal fun SectionAlignment.toGravity(): Int {
            return when (this) {
                START -> Gravity.START
                CENTER -> Gravity.CENTER
                END -> Gravity.END
            }
        }
    }
}

/** The Second Layer Settings. */
data class SecondLayerStyleSettings(
    /** The button layout. */
    val buttonLayout: ButtonLayout? = null,
    /** Show/hide the close button at the top-right corner of the Banner. If you are opening the Banner from your App's settings. We always recommend to enable a close button, so that users can easily dismiss the Banner. By default, close button is hidden. */
    val showCloseButton: Boolean? = null,
)
