package com.usercentrics.sdk

import android.content.Context
import android.content.DialogInterface
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.view.View
import android.view.ViewGroup
import android.view.Window
import android.view.WindowManager
import android.widget.FrameLayout
import android.widget.LinearLayout
import androidx.appcompat.app.AlertDialog
import com.usercentrics.sdk.ui.R

object PredefinedUIAlertDialogFactory {

    fun create(
        context: Context,
        rootView: View,
        cancelable: Boolean,
        statusBarIsCustomized: Boolean = false,
        fullscreen: Boolean = false,
        onDismissCallback: () -> Unit
    ): AlertDialog {

        val themeResId = if (fullscreen) {
            R.style.UsercentricsFullscreenBanner
        } else {
            R.style.UsercentricsBanner
        }

        val builder = AlertDialog.Builder(context, themeResId).apply {
            setCancelable(cancelable)
            setOnDismissListener { onDismissCallback() }
            setView(rootView)
        }

        val alertDialog = builder.create().apply {
            setOnShowListener(onShowListener(this, fullscreen))
            show()
            setLayoutParamsRecursively(rootView)

            window?.apply {
                setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
                setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                setDimAmount(0f)

                if (!fullscreen && statusBarIsCustomized) {
                    addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                }
            }
        }

        return alertDialog
    }

    private fun setLayoutParamsRecursively(view: View?) {
        if (view == null) {
            return
        }
        when (view.parent) {
            is FrameLayout -> {
                view.layoutParams = FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
            }

            is LinearLayout -> {
                view.layoutParams = LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
            }
        }
        setLayoutParamsRecursively(view.parent as? View)
    }

    private fun onShowListener(dialog: AlertDialog, isFullScreen: Boolean): DialogInterface.OnShowListener {
        return DialogInterface.OnShowListener {
            val bannerContainer = dialog.findViewById<ViewGroup>(R.id.ucBannerContainer)

            val viewTagTranslatesToGravity = bannerContainer?.tag as? Int ?: -1
            if (viewTagTranslatesToGravity < 0) {
                return@OnShowListener
            }

            val actualContent = bannerContainer?.getChildAt(0)
            actualContent?.post {
                if (isFullScreen) {
                    fullScreenMode(dialog, actualContent, viewTagTranslatesToGravity)

                } else {
                    safeAreScreenMode(dialog, actualContent, viewTagTranslatesToGravity)
                }
            }
        }
    }

    private fun safeAreScreenMode(
        dialog: AlertDialog,
        actualContent: View,
        viewTagTranslatesToGravity: Int
    ) {
        val displayMetrics = dialog.context.resources.displayMetrics
        val screenHeight = displayMetrics.heightPixels
        val maxHeight = (screenHeight / 1.25).toInt()

        if (actualContent.height > maxHeight) {
            actualContent.layoutParams = (actualContent.layoutParams as FrameLayout.LayoutParams).apply {
                height = maxHeight
                gravity = viewTagTranslatesToGravity
            }
        }
    }

    private fun fullScreenMode(
        dialog: AlertDialog,
        actualContent: View,
        viewTagTranslatesToGravity: Int
    ) {
        dialog.window?.apply {
            setFlags(
                WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS,
                WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS,
            )

            // Ensure content extends into the notch or cutout areas
            if (isAndroidPieOrHigher()) {
                attributes = attributes.apply {
                    layoutInDisplayCutoutMode = WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
                }
            }

            applyEdgeToEdgeUI(this)
        }

        // Adjust the content layout parameters to match the full screen
        actualContent.layoutParams = FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ).apply {
            gravity = viewTagTranslatesToGravity
        }

        // Apply safe insets
        applySafeInsets(actualContent)
    }

    // Adjust system UI visibility for edge-to-edge content
    @Suppress("DEPRECATION")
    private fun applyEdgeToEdgeUI(window: Window) {
        // For Android 11 or greater (API 30+)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            val controller = window.insetsController
            controller?.apply {
                hide(android.view.WindowInsets.Type.systemBars())
                systemBarsBehavior = android.view.WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            }
        } else {
            window.decorView.systemUiVisibility = (View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                    View.SYSTEM_UI_FLAG_HIDE_NAVIGATION or
                    View.SYSTEM_UI_FLAG_FULLSCREEN
                    )
        }
    }

    private fun applySafeInsets(view: View) {
        if (isAndroidPieOrHigher()) {
            view.setOnApplyWindowInsetsListener { v, insets ->
                val cutout = insets.displayCutout

                if (cutout != null) {
                    v.setPadding(
                        cutout.safeInsetLeft,
                        cutout.safeInsetTop,
                        cutout.safeInsetRight,
                        cutout.safeInsetBottom
                    )
                }
                insets
            }
        } else {
            view.setPadding(0, 0, 0, 0)
        }

        view.requestApplyInsets()
    }

    private fun isAndroidPieOrHigher(): Boolean {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.P
    }
}
